﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class RemoveTargetsRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API RemoveTargetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RemoveTargets"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the rule.</p>
   */
  inline const Aws::String& GetRule() const { return m_rule; }
  inline bool RuleHasBeenSet() const { return m_ruleHasBeenSet; }
  template <typename RuleT = Aws::String>
  void SetRule(RuleT&& value) {
    m_ruleHasBeenSet = true;
    m_rule = std::forward<RuleT>(value);
  }
  template <typename RuleT = Aws::String>
  RemoveTargetsRequest& WithRule(RuleT&& value) {
    SetRule(std::forward<RuleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the event bus associated with the rule. If you omit this,
   * the default event bus is used.</p>
   */
  inline const Aws::String& GetEventBusName() const { return m_eventBusName; }
  inline bool EventBusNameHasBeenSet() const { return m_eventBusNameHasBeenSet; }
  template <typename EventBusNameT = Aws::String>
  void SetEventBusName(EventBusNameT&& value) {
    m_eventBusNameHasBeenSet = true;
    m_eventBusName = std::forward<EventBusNameT>(value);
  }
  template <typename EventBusNameT = Aws::String>
  RemoveTargetsRequest& WithEventBusName(EventBusNameT&& value) {
    SetEventBusName(std::forward<EventBusNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the targets to remove from the rule.</p>
   */
  inline const Aws::Vector<Aws::String>& GetIds() const { return m_ids; }
  inline bool IdsHasBeenSet() const { return m_idsHasBeenSet; }
  template <typename IdsT = Aws::Vector<Aws::String>>
  void SetIds(IdsT&& value) {
    m_idsHasBeenSet = true;
    m_ids = std::forward<IdsT>(value);
  }
  template <typename IdsT = Aws::Vector<Aws::String>>
  RemoveTargetsRequest& WithIds(IdsT&& value) {
    SetIds(std::forward<IdsT>(value));
    return *this;
  }
  template <typename IdsT = Aws::String>
  RemoveTargetsRequest& AddIds(IdsT&& value) {
    m_idsHasBeenSet = true;
    m_ids.emplace_back(std::forward<IdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If this is a managed rule, created by an Amazon Web Services service on your
   * behalf, you must specify <code>Force</code> as <code>True</code> to remove
   * targets. This parameter is ignored for rules that are not managed rules. You can
   * check whether a rule is a managed rule by using <code>DescribeRule</code> or
   * <code>ListRules</code> and checking the <code>ManagedBy</code> field of the
   * response.</p>
   */
  inline bool GetForce() const { return m_force; }
  inline bool ForceHasBeenSet() const { return m_forceHasBeenSet; }
  inline void SetForce(bool value) {
    m_forceHasBeenSet = true;
    m_force = value;
  }
  inline RemoveTargetsRequest& WithForce(bool value) {
    SetForce(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_rule;

  Aws::String m_eventBusName;

  Aws::Vector<Aws::String> m_ids;

  bool m_force{false};
  bool m_ruleHasBeenSet = false;
  bool m_eventBusNameHasBeenSet = false;
  bool m_idsHasBeenSet = false;
  bool m_forceHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
