# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      module AnalyzedItems
        class AnalyzedMergeRequest
          CODE_DIRECTORIES = %w[app lib config keeps scripts db].freeze
          ADDITIONS_THRESHOLD = 5
          CODE_FILES_EXT = "rb|js|vue"
          SPEC_FILES_EXT = "rb|js"
          DOC_FILES_EXT = "tmpl|yaml|yml|md"

          def initialize(merge_request:, token:, project:, group:, dry_run:)
            @merge_request = merge_request
            @token = token
            @project = project
            @group = group
            @dry_run = dry_run
          end

          def analyze
            @files_with_missing_specs ||= fetch_files_with_missing_specs
          end

          def result
            {
              merge_request_iid: merge_request.iid,
              merge_request_web_url: merge_request.web_url,
              files_with_missing_specs: files_with_missing_specs,
              has_docs: has_docs?,
              has_feature_specs: has_feature_specs?,
              has_e2e_specs: has_e2e_specs?,
              added_feature_flag: added_feature_flag?
            }
          end

          private

          attr_reader :merge_request, :token, :project, :group, :dry_run, :files_with_missing_specs

          def merge_request_client
            @merge_request_client ||= (dry_run ? GitlabClient::MergeRequestsDryClient : GitlabClient::MergeRequestsClient).new(token: token, project: project)
          end

          def has_docs?
            doc_diffs.any?
          end

          def has_feature_specs?
            spec_diffs('features/').map { |diff| has_significant_addition?(diff) }.any?
          end

          def added_feature_flag?
            filter_diffs(%r{^(ee/)?config/feature_flags/.*\.yml$}).any?(&:new_file)
          end

          def has_e2e_specs?
            qa_spec_diffs.map { |diff| has_significant_addition?(diff) }.any?
          end

          def fetch_files_with_missing_specs
            code_paths_with_missing_specs = []

            CODE_DIRECTORIES.each do |dir|
              code_diffs = diffs_for_dir(dir, CODE_FILES_EXT)
              sp_diffs = spec_diffs
              code_paths_with_missing_specs << collect_paths_with_missing_specs(code_diffs, sp_diffs)
            end

            code_paths_with_missing_specs.flatten
          end

          def collect_paths_with_missing_specs(code_diffs, sp_diffs)
            result_array = []
            code_diffs.each do |code_diff|
              result_array << code_diff.new_path if has_significant_addition?(code_diff) && !skip_file?(code_diff.new_path) && !has_matching_spec_for_code?(code_diff, sp_diffs)
            end

            result_array
          end

          def has_matching_spec_for_code?(code_diff, sp_diffs)
            spec_file_names = get_spec_file_names(code_diff.new_path)
            sp_diffs.any? { |diff| ends_with_names?(diff.new_path, spec_file_names) }
          end

          def diffs
            @diffs ||= merge_request_client.merge_request_diffs(merge_request_iid: merge_request.iid)
          end

          def has_significant_addition?(diff_obj)
            content_lines = diff_obj.diff.split("\n")

            additions = content_lines.count { |line| line.start_with?("+") }
            subtractions = content_lines.count { |line| line.start_with?("-") }

            # Return true if additions are above threshold and there are equal or more additions than subtractions
            additions >= ADDITIONS_THRESHOLD && additions >= subtractions
          end

          def filter_diffs(pattern)
            diffs.select { |diff| diff.new_path =~ pattern }
          end

          def diffs_for_dir(dir, ext)
            filter_diffs(%r{^(ee/#{Regexp.escape(dir)}|#{Regexp.escape(dir)})/.*\.(#{ext})$})
          end

          def spec_diffs(spec_sub_dir = '')
            filter_diffs(%r{^(spec|ee/spec)/#{spec_sub_dir}.+_spec\.(#{SPEC_FILES_EXT})$})
          end

          def qa_spec_diffs
            filter_diffs(%r{^qa/qa/specs/features/.+(_spec|_shared_examples|_shared_context)\.rb$})
          end

          def doc_diffs
            filter_diffs(%r{^doc/.+\.(#{DOC_FILES_EXT})$}o)
          end

          def get_spec_file_names(path)
            filename = File.basename(path, ".*")

            %W[#{filename}_spec #{filename}_shared_examples]
          end

          def ends_with_names?(path, target_names)
            [File.basename(path, File.extname(path))].intersect?(target_names)
          end

          def skip_file?(path)
            path.end_with?('index.js')
          end
        end
      end
    end
  end
end
