/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.kinesis;

import com.amazonaws.services.kinesis.clientlibrary.lib.worker.InitialPositionInStream;
import com.amazonaws.services.kinesis.model.ExpiredIteratorException;
import com.amazonaws.services.kinesis.model.Shard;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.beam.sdk.io.kinesis.GetKinesisRecordsResult;
import org.apache.beam.sdk.io.kinesis.KinesisRecord;
import org.apache.beam.sdk.io.kinesis.KinesisShardClosedException;
import org.apache.beam.sdk.io.kinesis.RecordFilter;
import org.apache.beam.sdk.io.kinesis.ShardCheckpoint;
import org.apache.beam.sdk.io.kinesis.SimplifiedKinesisClient;
import org.apache.beam.sdk.io.kinesis.StartingPoint;
import org.apache.beam.sdk.io.kinesis.TransientKinesisException;
import org.apache.beam.sdk.io.kinesis.WatermarkPolicy;
import org.apache.beam.sdk.io.kinesis.WatermarkPolicyFactory;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Preconditions;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ShardRecordsIterator {
    private static final Logger LOG = LoggerFactory.getLogger(ShardRecordsIterator.class);
    private final SimplifiedKinesisClient kinesis;
    private final RecordFilter filter;
    private final String streamName;
    private final String shardId;
    private final AtomicReference<ShardCheckpoint> checkpoint;
    private final WatermarkPolicy watermarkPolicy;
    private final WatermarkPolicyFactory watermarkPolicyFactory;
    private final WatermarkPolicy latestRecordTimestampPolicy = WatermarkPolicyFactory.withArrivalTimePolicy().createWatermarkPolicy();
    private String shardIterator;

    ShardRecordsIterator(ShardCheckpoint initialCheckpoint, SimplifiedKinesisClient simplifiedKinesisClient, WatermarkPolicyFactory watermarkPolicyFactory) throws TransientKinesisException {
        this(initialCheckpoint, simplifiedKinesisClient, watermarkPolicyFactory, new RecordFilter());
    }

    ShardRecordsIterator(ShardCheckpoint initialCheckpoint, SimplifiedKinesisClient simplifiedKinesisClient, WatermarkPolicyFactory watermarkPolicyFactory, RecordFilter filter) throws TransientKinesisException {
        this.checkpoint = new AtomicReference<ShardCheckpoint>((ShardCheckpoint)Preconditions.checkNotNull((Object)initialCheckpoint, (Object)"initialCheckpoint"));
        this.filter = (RecordFilter)Preconditions.checkNotNull((Object)filter, (Object)"filter");
        this.kinesis = (SimplifiedKinesisClient)Preconditions.checkNotNull((Object)simplifiedKinesisClient, (Object)"simplifiedKinesisClient");
        this.streamName = initialCheckpoint.getStreamName();
        this.shardId = initialCheckpoint.getShardId();
        this.shardIterator = initialCheckpoint.getShardIterator(this.kinesis);
        this.watermarkPolicy = watermarkPolicyFactory.createWatermarkPolicy();
        this.watermarkPolicyFactory = watermarkPolicyFactory;
    }

    List<KinesisRecord> readNextBatch() throws TransientKinesisException, KinesisShardClosedException {
        if (this.shardIterator == null) {
            throw new KinesisShardClosedException(String.format("Shard iterator reached end of the shard: streamName=%s, shardId=%s", this.streamName, this.shardId));
        }
        GetKinesisRecordsResult response = this.fetchRecords();
        LOG.debug("Fetched {} new records from shard: streamName={}, shardId={}", new Object[]{response.getRecords().size(), this.streamName, this.shardId});
        List<KinesisRecord> filteredRecords = this.filter.apply(response.getRecords(), this.checkpoint.get());
        return filteredRecords;
    }

    private GetKinesisRecordsResult fetchRecords() throws TransientKinesisException {
        try {
            GetKinesisRecordsResult response = this.kinesis.getRecords(this.shardIterator, this.streamName, this.shardId);
            this.shardIterator = response.getNextShardIterator();
            return response;
        }
        catch (ExpiredIteratorException e) {
            LOG.info("Refreshing expired iterator for shard: streamName={}, shardId={}", new Object[]{this.streamName, this.shardId, e});
            this.shardIterator = this.checkpoint.get().getShardIterator(this.kinesis);
            return this.fetchRecords();
        }
    }

    ShardCheckpoint getCheckpoint() {
        return this.checkpoint.get();
    }

    void ackRecord(KinesisRecord record) {
        this.checkpoint.set(this.checkpoint.get().moveAfter(record));
        this.watermarkPolicy.update(record);
        this.latestRecordTimestampPolicy.update(record);
    }

    Instant getShardWatermark() {
        return this.watermarkPolicy.getWatermark();
    }

    Instant getLatestRecordTimestamp() {
        return this.latestRecordTimestampPolicy.getWatermark();
    }

    String getShardId() {
        return this.shardId;
    }

    String getStreamName() {
        return this.streamName;
    }

    List<ShardRecordsIterator> findSuccessiveShardRecordIterators() throws TransientKinesisException {
        List<Shard> shards = this.kinesis.listShardsFollowingClosedShard(this.streamName, this.shardId);
        ArrayList<ShardRecordsIterator> successiveShardRecordIterators = new ArrayList<ShardRecordsIterator>();
        for (Shard shard : shards) {
            if (!this.shardId.equals(shard.getParentShardId())) continue;
            ShardCheckpoint shardCheckpoint = new ShardCheckpoint(this.streamName, shard.getShardId(), new StartingPoint(InitialPositionInStream.TRIM_HORIZON));
            successiveShardRecordIterators.add(new ShardRecordsIterator(shardCheckpoint, this.kinesis, this.watermarkPolicyFactory));
        }
        return successiveShardRecordIterators;
    }
}

