/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.storage.s3;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsV2Request;
import com.amazonaws.services.s3.model.ListObjectsV2Result;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.StorageClass;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.inject.Inject;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.Map;
import org.apache.druid.java.util.common.IOE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.MapUtils;
import org.apache.druid.java.util.common.RetryUtils;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.segment.loading.DataSegmentMover;
import org.apache.druid.segment.loading.DataSegmentPusher;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.storage.s3.S3DataSegmentPusherConfig;
import org.apache.druid.storage.s3.S3Utils;
import org.apache.druid.storage.s3.ServerSideEncryptingAmazonS3;
import org.apache.druid.timeline.DataSegment;

public class S3DataSegmentMover
implements DataSegmentMover {
    private static final Logger log = new Logger(S3DataSegmentMover.class);
    private final Supplier<ServerSideEncryptingAmazonS3> s3ClientSupplier;
    private final S3DataSegmentPusherConfig config;

    @Inject
    public S3DataSegmentMover(Supplier<ServerSideEncryptingAmazonS3> s3ClientSupplier, S3DataSegmentPusherConfig config) {
        this.s3ClientSupplier = s3ClientSupplier;
        this.config = config;
    }

    public DataSegment move(DataSegment segment, Map<String, Object> targetLoadSpec) throws SegmentLoadingException {
        try {
            String targetS3Path;
            Map loadSpec = segment.getLoadSpec();
            String s3Bucket = MapUtils.getString((Map)loadSpec, (String)"bucket");
            String s3Path = MapUtils.getString((Map)loadSpec, (String)"key");
            String targetS3Bucket = MapUtils.getString(targetLoadSpec, (String)"bucket");
            String targetS3BaseKey = MapUtils.getString(targetLoadSpec, (String)"baseKey");
            if (s3Path.endsWith("/")) {
                ListObjectsV2Result list = ((ServerSideEncryptingAmazonS3)this.s3ClientSupplier.get()).listObjectsV2(new ListObjectsV2Request().withBucketName(s3Bucket).withPrefix(s3Path + "/"));
                targetS3Path = S3Utils.constructSegmentBasePath(targetS3BaseKey, DataSegmentPusher.getDefaultStorageDir((DataSegment)segment, (boolean)false));
                for (S3ObjectSummary objectSummary : list.getObjectSummaries()) {
                    String fileName = Paths.get(objectSummary.getKey(), new String[0]).getFileName().toString();
                    if (targetS3Bucket.isEmpty()) {
                        throw new SegmentLoadingException("Target S3 bucket is not specified", new Object[0]);
                    }
                    if (targetS3Path.isEmpty()) {
                        throw new SegmentLoadingException("Target S3 baseKey is not specified", new Object[0]);
                    }
                    this.safeMove(s3Bucket, s3Path, targetS3Bucket, targetS3Path + "/" + fileName);
                }
            } else {
                targetS3Path = S3Utils.constructSegmentPath(targetS3BaseKey, DataSegmentPusher.getDefaultStorageDir((DataSegment)segment, (boolean)false));
                if (targetS3Bucket.isEmpty()) {
                    throw new SegmentLoadingException("Target S3 bucket is not specified", new Object[0]);
                }
                if (targetS3Path.isEmpty()) {
                    throw new SegmentLoadingException("Target S3 baseKey is not specified", new Object[0]);
                }
                this.safeMove(s3Bucket, s3Path, targetS3Bucket, targetS3Path);
            }
            return segment.withLoadSpec((Map)ImmutableMap.builder().putAll(Maps.filterKeys((Map)loadSpec, (Predicate)new Predicate<String>(){

                public boolean apply(String input) {
                    return !"bucket".equals(input) && !"key".equals(input);
                }
            })).put((Object)"bucket", (Object)targetS3Bucket).put((Object)"key", (Object)targetS3Path).build());
        }
        catch (AmazonServiceException e) {
            throw new SegmentLoadingException((Throwable)e, "Unable to move segment[%s]: [%s]", new Object[]{segment.getId(), e});
        }
    }

    private void safeMove(String s3Bucket, String s3Path, String targetS3Bucket, String targetS3Path) throws SegmentLoadingException {
        try {
            S3Utils.retryS3Operation(() -> {
                String copyMsg = StringUtils.format((String)"[s3://%s/%s] to [s3://%s/%s]", (Object[])new Object[]{s3Bucket, s3Path, targetS3Bucket, targetS3Path});
                try {
                    this.selfCheckingMove(s3Bucket, targetS3Bucket, s3Path, targetS3Path, copyMsg);
                    return null;
                }
                catch (AmazonServiceException | IOException | SegmentLoadingException e) {
                    log.info(e, "Error while trying to move " + copyMsg, new Object[0]);
                    throw e;
                }
            });
        }
        catch (Exception e) {
            Throwables.propagateIfInstanceOf((Throwable)e, AmazonServiceException.class);
            Throwables.propagateIfInstanceOf((Throwable)e, SegmentLoadingException.class);
            throw new RuntimeException(e);
        }
    }

    private void selfCheckingMove(String s3Bucket, String targetS3Bucket, String s3Path, String targetS3Path, String copyMsg) throws IOException, SegmentLoadingException {
        if (s3Bucket.equals(targetS3Bucket) && s3Path.equals(targetS3Path)) {
            log.info("No need to move file[s3://%s/%s] onto itself", new Object[]{s3Bucket, s3Path});
            return;
        }
        ServerSideEncryptingAmazonS3 s3Client = (ServerSideEncryptingAmazonS3)this.s3ClientSupplier.get();
        if (s3Client.doesObjectExist(s3Bucket, s3Path)) {
            ListObjectsV2Result listResult = s3Client.listObjectsV2(new ListObjectsV2Request().withBucketName(s3Bucket).withPrefix(s3Path).withMaxKeys(Integer.valueOf(1)));
            if (listResult.getObjectSummaries().size() == 0) {
                throw new ISE("Unable to list object [s3://%s/%s]", new Object[]{s3Bucket, s3Path});
            }
            S3ObjectSummary objectSummary = (S3ObjectSummary)listResult.getObjectSummaries().get(0);
            if (objectSummary.getStorageClass() != null && StorageClass.fromValue((String)StringUtils.toUpperCase((String)objectSummary.getStorageClass())).equals((Object)StorageClass.Glacier)) {
                throw new AmazonServiceException(StringUtils.format((String)"Cannot move file[s3://%s/%s] of storage class glacier, skipping.", (Object[])new Object[]{s3Bucket, s3Path}));
            }
            log.info("Moving file %s", new Object[]{copyMsg});
            CopyObjectRequest copyRequest = new CopyObjectRequest(s3Bucket, s3Path, targetS3Bucket, targetS3Path);
            if (!this.config.getDisableAcl()) {
                copyRequest.setAccessControlList(S3Utils.grantFullControlToBucketOwner(s3Client, targetS3Bucket));
            }
            s3Client.copyObject(copyRequest);
            if (!s3Client.doesObjectExist(targetS3Bucket, targetS3Path)) {
                throw new IOE("After copy was reported as successful the file doesn't exist in the target location [%s]", new Object[]{copyMsg});
            }
            this.deleteWithRetriesSilent(s3Bucket, s3Path);
            log.debug("Finished moving file %s", new Object[]{copyMsg});
        } else if (s3Client.doesObjectExist(targetS3Bucket, targetS3Path)) {
            log.info("Not moving file [s3://%s/%s], already present in target location [s3://%s/%s]", new Object[]{s3Bucket, s3Path, targetS3Bucket, targetS3Path});
        } else {
            throw new SegmentLoadingException("Unable to move file %s, not present in either source or target location", new Object[]{copyMsg});
        }
    }

    private void deleteWithRetriesSilent(String s3Bucket, String s3Path) {
        try {
            this.deleteWithRetries(s3Bucket, s3Path);
        }
        catch (Exception e) {
            log.error((Throwable)e, "Failed to delete file [s3://%s/%s], giving up", new Object[]{s3Bucket, s3Path});
        }
    }

    private void deleteWithRetries(String s3Bucket, String s3Path) throws Exception {
        RetryUtils.retry(() -> {
            try {
                ((ServerSideEncryptingAmazonS3)this.s3ClientSupplier.get()).deleteObject(s3Bucket, s3Path);
                return null;
            }
            catch (Exception e) {
                log.info((Throwable)e, "Error while trying to delete [s3://%s/%s]", new Object[]{s3Bucket, s3Path});
                throw e;
            }
        }, S3Utils.S3RETRY, (int)3);
    }
}

