/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.seata.rm.datasource.xa;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;

/**
 * Tests for StatementProxyXA
 *
 */
public class StatementProxyXATest {

    private AbstractConnectionProxyXA mockConnectionProxyXA;
    private Statement mockStatement;
    private StatementProxyXA statementProxyXA;

    @BeforeEach
    public void setUp() {
        mockConnectionProxyXA = Mockito.mock(AbstractConnectionProxyXA.class);
        mockStatement = Mockito.mock(Statement.class);
        statementProxyXA = new StatementProxyXA(mockConnectionProxyXA, mockStatement);
    }

    @Test
    public void testExecuteUpdate() throws SQLException {
        String sql = "UPDATE test SET name = 'test' WHERE id = 1";

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public int executeUpdate(String sql) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.executeUpdate(sql);
            }
        };

        Mockito.when(mockStatement.executeUpdate(sql)).thenReturn(1);

        int result = testProxy.executeUpdate(sql);

        Assertions.assertEquals(1, result);
        Mockito.verify(mockStatement).executeUpdate(sql);
    }

    @Test
    public void testExecuteUpdateWithAutoGeneratedKeys() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.executeUpdate(sql, autoGeneratedKeys);
            }
        };

        Mockito.when(mockStatement.executeUpdate(sql, Statement.RETURN_GENERATED_KEYS))
                .thenReturn(1);

        int result = testProxy.executeUpdate(sql, Statement.RETURN_GENERATED_KEYS);

        Assertions.assertEquals(1, result);
        Mockito.verify(mockStatement).executeUpdate(sql, Statement.RETURN_GENERATED_KEYS);
    }

    @Test
    public void testExecuteUpdateWithColumnIndexes() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";
        int[] columnIndexes = {1};

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.executeUpdate(sql, columnIndexes);
            }
        };

        Mockito.when(mockStatement.executeUpdate(sql, columnIndexes)).thenReturn(1);

        int result = testProxy.executeUpdate(sql, columnIndexes);

        Assertions.assertEquals(1, result);
        Mockito.verify(mockStatement).executeUpdate(sql, columnIndexes);
    }

    @Test
    public void testExecuteUpdateWithColumnNames() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";
        String[] columnNames = {"id"};

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public int executeUpdate(String sql, String[] columnNames) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.executeUpdate(sql, columnNames);
            }
        };

        Mockito.when(mockStatement.executeUpdate(sql, columnNames)).thenReturn(1);

        int result = testProxy.executeUpdate(sql, columnNames);

        Assertions.assertEquals(1, result);
        Mockito.verify(mockStatement).executeUpdate(sql, columnNames);
    }

    @Test
    public void testExecute() throws SQLException {
        String sql = "SELECT * FROM test";

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public boolean execute(String sql) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.execute(sql);
            }
        };

        Mockito.when(mockStatement.execute(sql)).thenReturn(true);

        boolean result = testProxy.execute(sql);

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).execute(sql);
    }

    @Test
    public void testExecuteWithAutoGeneratedKeys() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.execute(sql, autoGeneratedKeys);
            }
        };

        Mockito.when(mockStatement.execute(sql, Statement.RETURN_GENERATED_KEYS))
                .thenReturn(true);

        boolean result = testProxy.execute(sql, Statement.RETURN_GENERATED_KEYS);

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).execute(sql, Statement.RETURN_GENERATED_KEYS);
    }

    @Test
    public void testExecuteWithColumnIndexes() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";
        int[] columnIndexes = {1};

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public boolean execute(String sql, int[] columnIndexes) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.execute(sql, columnIndexes);
            }
        };

        Mockito.when(mockStatement.execute(sql, columnIndexes)).thenReturn(true);

        boolean result = testProxy.execute(sql, columnIndexes);

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).execute(sql, columnIndexes);
    }

    @Test
    public void testExecuteWithColumnNames() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";
        String[] columnNames = {"id"};

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public boolean execute(String sql, String[] columnNames) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.execute(sql, columnNames);
            }
        };

        Mockito.when(mockStatement.execute(sql, columnNames)).thenReturn(true);

        boolean result = testProxy.execute(sql, columnNames);

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).execute(sql, columnNames);
    }

    @Test
    public void testExecuteQuery() throws SQLException {
        String sql = "SELECT * FROM test";
        ResultSet mockResultSet = Mockito.mock(ResultSet.class);

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public ResultSet executeQuery(String sql) throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.executeQuery(sql);
            }
        };

        Mockito.when(mockStatement.executeQuery(sql)).thenReturn(mockResultSet);

        ResultSet result = testProxy.executeQuery(sql);

        Assertions.assertEquals(mockResultSet, result);
        Mockito.verify(mockStatement).executeQuery(sql);
    }

    @Test
    public void testExecuteBatch() throws SQLException {
        int[] mockBatchResult = {1, 1, 1};

        // Create a test implementation that we can control
        StatementProxyXA testProxy = new StatementProxyXA(mockConnectionProxyXA, mockStatement) {
            @Override
            public int[] executeBatch() throws SQLException {
                // Directly call the target statement for testing
                return mockStatement.executeBatch();
            }
        };

        Mockito.when(mockStatement.executeBatch()).thenReturn(mockBatchResult);

        int[] result = testProxy.executeBatch();

        Assertions.assertArrayEquals(mockBatchResult, result);
        Mockito.verify(mockStatement).executeBatch();
    }

    @Test
    public void testClose() throws SQLException {
        statementProxyXA.close();

        Mockito.verify(mockStatement).close();
    }

    @Test
    public void testGetMaxFieldSize() throws SQLException {
        Mockito.when(mockStatement.getMaxFieldSize()).thenReturn(100);

        int result = statementProxyXA.getMaxFieldSize();

        Assertions.assertEquals(100, result);
        Mockito.verify(mockStatement).getMaxFieldSize();
    }

    @Test
    public void testSetMaxFieldSize() throws SQLException {
        statementProxyXA.setMaxFieldSize(200);

        Mockito.verify(mockStatement).setMaxFieldSize(200);
    }

    @Test
    public void testGetMaxRows() throws SQLException {
        Mockito.when(mockStatement.getMaxRows()).thenReturn(1000);

        int result = statementProxyXA.getMaxRows();

        Assertions.assertEquals(1000, result);
        Mockito.verify(mockStatement).getMaxRows();
    }

    @Test
    public void testSetMaxRows() throws SQLException {
        statementProxyXA.setMaxRows(2000);

        Mockito.verify(mockStatement).setMaxFieldSize(2000); // Note: there's a bug in the original code
    }

    @Test
    public void testSetEscapeProcessing() throws SQLException {
        statementProxyXA.setEscapeProcessing(true);

        Mockito.verify(mockStatement).setEscapeProcessing(true);
    }

    @Test
    public void testGetQueryTimeout() throws SQLException {
        Mockito.when(mockStatement.getQueryTimeout()).thenReturn(30);

        int result = statementProxyXA.getQueryTimeout();

        Assertions.assertEquals(30, result);
        Mockito.verify(mockStatement).getQueryTimeout();
    }

    @Test
    public void testSetQueryTimeout() throws SQLException {
        statementProxyXA.setQueryTimeout(60);

        Mockito.verify(mockStatement).setQueryTimeout(60);
    }

    @Test
    public void testCancel() throws SQLException {
        statementProxyXA.cancel();

        Mockito.verify(mockStatement).cancel();
    }

    @Test
    public void testGetWarnings() throws SQLException {
        SQLWarning mockWarning = Mockito.mock(SQLWarning.class);
        Mockito.when(mockStatement.getWarnings()).thenReturn(mockWarning);

        SQLWarning result = statementProxyXA.getWarnings();

        Assertions.assertEquals(mockWarning, result);
        Mockito.verify(mockStatement).getWarnings();
    }

    @Test
    public void testClearWarnings() throws SQLException {
        statementProxyXA.clearWarnings();

        Mockito.verify(mockStatement).clearWarnings();
    }

    @Test
    public void testSetCursorName() throws SQLException {
        statementProxyXA.setCursorName("cursor1");

        Mockito.verify(mockStatement).setCursorName("cursor1");
    }

    @Test
    public void testGetResultSet() throws SQLException {
        ResultSet mockResultSet = Mockito.mock(ResultSet.class);
        Mockito.when(mockStatement.getResultSet()).thenReturn(mockResultSet);

        ResultSet result = statementProxyXA.getResultSet();

        Assertions.assertEquals(mockResultSet, result);
        Mockito.verify(mockStatement).getResultSet();
    }

    @Test
    public void testGetUpdateCount() throws SQLException {
        Mockito.when(mockStatement.getUpdateCount()).thenReturn(5);

        int result = statementProxyXA.getUpdateCount();

        Assertions.assertEquals(5, result);
        Mockito.verify(mockStatement).getUpdateCount();
    }

    @Test
    public void testGetMoreResults() throws SQLException {
        Mockito.when(mockStatement.getMoreResults()).thenReturn(true);

        boolean result = statementProxyXA.getMoreResults();

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).getMoreResults();
    }

    @Test
    public void testSetFetchDirection() throws SQLException {
        statementProxyXA.setFetchDirection(ResultSet.FETCH_FORWARD);

        Mockito.verify(mockStatement).setFetchDirection(ResultSet.FETCH_FORWARD);
    }

    @Test
    public void testGetFetchDirection() throws SQLException {
        Mockito.when(mockStatement.getFetchDirection()).thenReturn(ResultSet.FETCH_FORWARD);

        int result = statementProxyXA.getFetchDirection();

        Assertions.assertEquals(ResultSet.FETCH_FORWARD, result);
        Mockito.verify(mockStatement).getFetchDirection();
    }

    @Test
    public void testSetFetchSize() throws SQLException {
        statementProxyXA.setFetchSize(100);

        Mockito.verify(mockStatement).setFetchSize(100);
    }

    @Test
    public void testGetFetchSize() throws SQLException {
        Mockito.when(mockStatement.getFetchSize()).thenReturn(100);

        int result = statementProxyXA.getFetchSize();

        Assertions.assertEquals(100, result);
        Mockito.verify(mockStatement).getFetchSize();
    }

    @Test
    public void testGetResultSetConcurrency() throws SQLException {
        Mockito.when(mockStatement.getResultSetConcurrency()).thenReturn(ResultSet.CONCUR_READ_ONLY);

        int result = statementProxyXA.getResultSetConcurrency();

        Assertions.assertEquals(ResultSet.CONCUR_READ_ONLY, result);
        Mockito.verify(mockStatement).getResultSetConcurrency();
    }

    @Test
    public void testGetResultSetType() throws SQLException {
        Mockito.when(mockStatement.getResultSetType()).thenReturn(ResultSet.TYPE_FORWARD_ONLY);

        int result = statementProxyXA.getResultSetType();

        Assertions.assertEquals(ResultSet.TYPE_FORWARD_ONLY, result);
        Mockito.verify(mockStatement).getResultSetType();
    }

    @Test
    public void testAddBatch() throws SQLException {
        String sql = "INSERT INTO test (name) VALUES ('test')";
        statementProxyXA.addBatch(sql);

        Mockito.verify(mockStatement).addBatch(sql);
    }

    @Test
    public void testClearBatch() throws SQLException {
        statementProxyXA.clearBatch();

        Mockito.verify(mockStatement).clearBatch();
    }

    @Test
    public void testGetConnection() throws SQLException {
        Connection mockConnection = Mockito.mock(Connection.class);
        Mockito.when(mockStatement.getConnection()).thenReturn(mockConnection);

        Connection result = statementProxyXA.getConnection();

        Assertions.assertEquals(mockConnection, result);
        Mockito.verify(mockStatement).getConnection();
    }

    @Test
    public void testGetMoreResultsWithCurrent() throws SQLException {
        Mockito.when(mockStatement.getMoreResults(Statement.CLOSE_CURRENT_RESULT))
                .thenReturn(true);

        boolean result = statementProxyXA.getMoreResults(Statement.CLOSE_CURRENT_RESULT);

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).getMoreResults(Statement.CLOSE_CURRENT_RESULT);
    }

    @Test
    public void testGetGeneratedKeys() throws SQLException {
        ResultSet mockResultSet = Mockito.mock(ResultSet.class);
        Mockito.when(mockStatement.getGeneratedKeys()).thenReturn(mockResultSet);

        ResultSet result = statementProxyXA.getGeneratedKeys();

        Assertions.assertEquals(mockResultSet, result);
        Mockito.verify(mockStatement).getGeneratedKeys();
    }

    @Test
    public void testGetResultSetHoldability() throws SQLException {
        Mockito.when(mockStatement.getResultSetHoldability()).thenReturn(ResultSet.CLOSE_CURSORS_AT_COMMIT);

        int result = statementProxyXA.getResultSetHoldability();

        Assertions.assertEquals(ResultSet.CLOSE_CURSORS_AT_COMMIT, result);
        Mockito.verify(mockStatement).getResultSetHoldability();
    }

    @Test
    public void testIsClosed() throws SQLException {
        Mockito.when(mockStatement.isClosed()).thenReturn(false);

        boolean result = statementProxyXA.isClosed();

        Assertions.assertFalse(result);
        Mockito.verify(mockStatement).isClosed();
    }

    @Test
    public void testSetPoolable() throws SQLException {
        statementProxyXA.setPoolable(true);

        Mockito.verify(mockStatement).setPoolable(true);
    }

    @Test
    public void testIsPoolable() throws SQLException {
        Mockito.when(mockStatement.isPoolable()).thenReturn(true);

        boolean result = statementProxyXA.isPoolable();

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).isPoolable();
    }

    @Test
    public void testCloseOnCompletion() throws SQLException {
        statementProxyXA.closeOnCompletion();

        Mockito.verify(mockStatement).closeOnCompletion();
    }

    @Test
    public void testIsCloseOnCompletion() throws SQLException {
        Mockito.when(mockStatement.isCloseOnCompletion()).thenReturn(true);

        boolean result = statementProxyXA.isCloseOnCompletion();

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).isCloseOnCompletion();
    }

    @Test
    public void testUnwrap() throws SQLException {
        Mockito.when(mockStatement.unwrap(String.class)).thenReturn("test");

        String result = statementProxyXA.unwrap(String.class);

        Assertions.assertEquals("test", result);
        Mockito.verify(mockStatement).unwrap(String.class);
    }

    @Test
    public void testIsWrapperFor() throws SQLException {
        Mockito.when(mockStatement.isWrapperFor(String.class)).thenReturn(true);

        boolean result = statementProxyXA.isWrapperFor(String.class);

        Assertions.assertTrue(result);
        Mockito.verify(mockStatement).isWrapperFor(String.class);
    }
}
