/*
 * Decompiled with CFR 0.152.
 */
package org.apache.inlong.tubemq.server.broker.msgstore.disk;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.inlong.tubemq.corebase.utils.CheckSum;
import org.apache.inlong.tubemq.corebase.utils.ServiceStatusHolder;
import org.apache.inlong.tubemq.server.broker.msgstore.disk.Segment;
import org.apache.inlong.tubemq.server.broker.msgstore.disk.SegmentType;
import org.apache.inlong.tubemq.server.broker.stats.BrokerSrvStatsHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSegment
implements Segment {
    private static final Logger logger = LoggerFactory.getLogger(FileSegment.class);
    private final long start;
    private final File file;
    private final RandomAccessFile randFile;
    private final FileChannel channel;
    private final AtomicLong cachedSize;
    private final AtomicLong flushedSize;
    private final SegmentType segmentType;
    private volatile boolean mutable = false;
    private long expiredTime = 0L;
    private final AtomicBoolean expired = new AtomicBoolean(false);
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final AtomicLong leftAppendTime = new AtomicLong(-2L);
    private final AtomicLong rightAppendTime = new AtomicLong(-2L);

    public FileSegment(long start, File file, SegmentType type) throws IOException {
        this(start, file, true, type, Long.MAX_VALUE);
    }

    public FileSegment(long start, File file, boolean mutable, SegmentType type) throws IOException {
        this(start, file, mutable, type, Long.MAX_VALUE);
    }

    public FileSegment(long start, File file, SegmentType type, long checkOffset) throws IOException {
        this(start, file, true, type, checkOffset);
    }

    private FileSegment(long start, File file, boolean mutable, SegmentType type, long checkOffset) throws IOException {
        this.segmentType = type;
        this.start = start;
        this.file = file;
        this.mutable = mutable;
        this.cachedSize = new AtomicLong(0L);
        this.flushedSize = new AtomicLong(0L);
        this.randFile = new RandomAccessFile(this.file, "rw");
        this.channel = this.randFile.getChannel();
        if (mutable) {
            long remaining;
            long startMs = System.currentTimeMillis();
            long l = remaining = checkOffset == Long.MAX_VALUE ? -1L : checkOffset - this.start;
            if (this.segmentType == SegmentType.DATA) {
                RecoverResult recoverResult = this.recoverData(remaining);
                if (recoverResult.isEqual()) {
                    logger.info("[File Store] Data Segment recover success, ignore content check!");
                } else if (recoverResult.getTruncated() > 0L) {
                    logger.info(new StringBuilder(512).append("[File Store] Recover DATA Segment succeeded in ").append((System.currentTimeMillis() - startMs) / 1000L).append(" seconds. ").append(recoverResult.getTruncated()).append(" bytes truncated.").toString());
                }
            } else {
                RecoverResult recoverResult = this.recoverIndex(remaining);
                if (recoverResult.isEqual()) {
                    logger.info("[File Store] Index Segment recover success, ignore content check!");
                } else if (recoverResult.getTruncated() > 0L) {
                    logger.info(new StringBuilder(512).append("[File Store] Recover Index Segment succeeded in ").append((System.currentTimeMillis() - startMs) / 1000L).append(" seconds. ").append(recoverResult.getTruncated()).append(" bytes truncated.").toString());
                }
            }
        } else {
            try {
                this.cachedSize.set(this.channel.size());
                this.flushedSize.set(this.cachedSize.get());
            }
            catch (Exception e) {
                if (e instanceof IOException) {
                    ServiceStatusHolder.addReadIOErrCnt();
                    BrokerSrvStatsHolder.incDiskIOExcCnt();
                }
                if (this.segmentType == SegmentType.DATA) {
                    logger.error("[File Store] Set DATA Segment cachedSize error", (Throwable)e);
                }
                logger.error("[File Store] Set INDEX Segment cachedSize error", (Throwable)e);
            }
        }
        if (this.segmentType == SegmentType.INDEX) {
            if (this.cachedSize.get() == 0L) {
                if (this.mutable) {
                    this.leftAppendTime.set(System.currentTimeMillis());
                    this.rightAppendTime.set(System.currentTimeMillis());
                }
            } else {
                this.leftAppendTime.set(this.getRecordTime(this.start));
                this.rightAppendTime.set(this.getRecordTime(this.start + this.cachedSize.get() - 28L));
            }
        }
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            try {
                if (this.channel.isOpen()) {
                    if (this.mutable) {
                        this.flush(true);
                    }
                    this.channel.close();
                }
                this.randFile.close();
            }
            catch (Throwable ee) {
                if (ee instanceof IOException) {
                    ServiceStatusHolder.addReadIOErrCnt();
                    BrokerSrvStatsHolder.incDiskIOExcCnt();
                }
                logger.error(new StringBuilder(512).append("[File Store] Close ").append(this.file.getAbsoluteFile().toString()).append("'s ").append((Object)this.segmentType).append(" file failure").toString(), ee);
            }
        }
    }

    @Override
    public void deleteFile() {
        this.closed.set(true);
        try {
            if (this.channel.isOpen()) {
                if (this.mutable) {
                    this.flush(true);
                }
                this.channel.close();
            }
            this.randFile.close();
        }
        catch (Throwable e1) {
            if (e1 instanceof IOException) {
                ServiceStatusHolder.addReadIOErrCnt();
                BrokerSrvStatsHolder.incDiskIOExcCnt();
            }
            logger.error("[File Store] failure to close channel ", e1);
        }
        try {
            logger.info(new StringBuilder(512).append("[File Store] delete file ").append(this.file.getAbsoluteFile()).toString());
            this.file.delete();
        }
        catch (Throwable ee) {
            if (ee instanceof IOException) {
                ServiceStatusHolder.addReadIOErrCnt();
                BrokerSrvStatsHolder.incDiskIOExcCnt();
            }
            logger.error("[File Store] failure to delete file ", ee);
        }
    }

    @Override
    public long append(ByteBuffer buf, long leftTime, long rightTime) throws IOException {
        if (!this.mutable) {
            if (this.segmentType == SegmentType.DATA) {
                throw new UnsupportedOperationException("[File Store] Data Segment is immutable!");
            }
            throw new UnsupportedOperationException("[File Store] Index Segment is immutable!");
        }
        if (this.closed.get()) {
            throw new UnsupportedOperationException("[File Store] Segment is closed!");
        }
        long offset = this.cachedSize.get();
        int sizeInBytes = 0;
        while (buf.hasRemaining()) {
            sizeInBytes += this.channel.write(buf);
        }
        this.cachedSize.addAndGet(sizeInBytes);
        if (this.segmentType == SegmentType.INDEX) {
            this.rightAppendTime.set(rightTime);
            if (offset == 0L) {
                this.leftAppendTime.set(leftTime);
            }
        }
        return this.start + offset;
    }

    @Override
    public long flush(boolean force) throws IOException {
        this.channel.force(force);
        this.flushedSize.set(this.cachedSize.get());
        return this.start + this.flushedSize.get();
    }

    @Override
    public boolean isExpired() {
        return this.expired.get();
    }

    @Override
    public boolean needDelete() {
        return this.expired.get() && System.currentTimeMillis() - this.expiredTime > 120000L;
    }

    @Override
    public boolean isClosed() {
        return this.closed.get();
    }

    @Override
    public boolean contains(long offset) {
        return this.getCachedSize() == 0L && offset == this.start || this.getCachedSize() > 0L && offset >= this.start && offset <= this.start + this.getCachedSize() - 1L;
    }

    @Override
    public void relViewRef() {
    }

    @Override
    public long getStart() {
        return this.start;
    }

    @Override
    public long getLast() {
        return this.start + this.cachedSize.get();
    }

    @Override
    public long getCommitLast() {
        return this.start + this.flushedSize.get();
    }

    @Override
    public boolean isMutable() {
        return this.mutable;
    }

    @Override
    public void setMutable(boolean mutable) {
        this.mutable = mutable;
    }

    @Override
    public long getLeftAppendTime() {
        return this.leftAppendTime.get();
    }

    @Override
    public long getRightAppendTime() {
        return this.rightAppendTime.get();
    }

    @Override
    public boolean containTime(long timestamp) {
        if (this.getCachedSize() == 0L) {
            return this.mutable;
        }
        if (timestamp >= this.leftAppendTime.get()) {
            if (this.mutable) {
                return true;
            }
            return timestamp <= this.rightAppendTime.get();
        }
        return false;
    }

    @Override
    public long getCachedSize() {
        return this.cachedSize.get();
    }

    @Override
    public long getCommitSize() {
        return this.flushedSize.get();
    }

    @Override
    public final File getFile() {
        return this.file;
    }

    @Override
    public void read(ByteBuffer bf, long absOffset) throws IOException {
        int l;
        if (this.isExpired()) {
            // empty if block
        }
        int size = 0;
        long startPos = absOffset - this.start;
        while (bf.hasRemaining() && (l = this.channel.read(bf, startPos + (long)size)) >= 0) {
            size += l;
        }
    }

    @Override
    public void relRead(ByteBuffer bf, long relOffset) throws IOException {
        int l;
        if (this.isExpired()) {
            // empty if block
        }
        int size = 0;
        while (bf.hasRemaining() && (l = this.channel.read(bf, relOffset + (long)size)) >= 0) {
            size += l;
        }
    }

    @Override
    public long getRecordTime(long reqOffset) throws IOException {
        int l;
        ByteBuffer readUnit = ByteBuffer.allocate(28);
        int size = 0;
        while (readUnit.hasRemaining() && (l = this.channel.read(readUnit, reqOffset - this.start + (long)size)) >= 0) {
            size += l;
        }
        readUnit.flip();
        return readUnit.getLong(20);
    }

    @Override
    public int checkAndSetExpired(long checkTimestamp, long maxValidTimeMs) {
        if (this.expired.get()) {
            return -1;
        }
        if (this.closed.get()) {
            return 0;
        }
        if (!this.mutable && checkTimestamp - this.file.lastModified() > maxValidTimeMs) {
            if (this.expired.compareAndSet(false, true)) {
                this.expiredTime = System.currentTimeMillis();
            }
            return 1;
        }
        return 0;
    }

    private RecoverResult recoverData(long checkOffset) throws IOException {
        if (!this.mutable) {
            throw new UnsupportedOperationException("[File Store] The Data Segment must be mutable!");
        }
        long totalBytes = this.channel.size();
        if (totalBytes == checkOffset || checkOffset == -1L) {
            this.cachedSize.set(totalBytes);
            this.flushedSize.set(totalBytes);
            this.channel.position(totalBytes);
            return new RecoverResult(0L, totalBytes == checkOffset);
        }
        long validBytes = 0L;
        long next = 0L;
        long itemRead = 0L;
        int itemMsglen = 0;
        int itemMsgToken = 0;
        int itemCheckSum = 0;
        long itemNext = 0L;
        ByteBuffer itemBuf = ByteBuffer.allocate(52);
        do {
            itemBuf.rewind();
            itemRead = this.channel.read(itemBuf);
            if (itemRead < 52L) {
                next = -1L;
            } else {
                itemBuf.flip();
                itemMsglen = itemBuf.getInt() - 48;
                itemMsgToken = itemBuf.getInt();
                itemCheckSum = itemBuf.getInt();
                itemNext = validBytes + 52L + (long)itemMsglen;
                if (itemMsgToken != 46766264 || itemMsglen <= 0 || itemMsglen > 0x1C00000 || itemNext > totalBytes) {
                    next = -1L;
                } else {
                    ByteBuffer messageBuffer = ByteBuffer.allocate(itemMsglen);
                    while (messageBuffer.hasRemaining()) {
                        itemRead = this.channel.read(messageBuffer);
                        if (itemRead >= 0L) continue;
                        throw new IOException("[File Store] The Data Segment is changing in recover processing!");
                    }
                    next = CheckSum.crc32((byte[])messageBuffer.array()) != itemCheckSum ? -1L : itemNext;
                }
            }
            if (next < 0L) continue;
            validBytes = next;
        } while (next >= 0L);
        if (totalBytes != validBytes) {
            this.channel.truncate(validBytes);
        }
        this.cachedSize.set(validBytes);
        this.flushedSize.set(validBytes);
        this.channel.position(validBytes);
        return new RecoverResult(totalBytes - validBytes, false);
    }

    private RecoverResult recoverIndex(long checkOffset) throws IOException {
        if (!this.mutable) {
            throw new UnsupportedOperationException("[File Store] The Index Segment must be mutable!");
        }
        long totalBytes = this.channel.size();
        if (totalBytes == checkOffset) {
            this.cachedSize.set(totalBytes);
            this.flushedSize.set(totalBytes);
            this.channel.position(totalBytes);
            return new RecoverResult(0L, true);
        }
        long validBytes = 0L;
        long next = 0L;
        long itemRead = 0L;
        int itemMsgPartId = 0;
        long itemMsgOffset = 0L;
        int itemMsglen = 0;
        int itemKeyCode = 0;
        long itemTimeRecv = 0L;
        long itemNext = 0L;
        ByteBuffer itemBuf = ByteBuffer.allocate(28);
        do {
            itemBuf.rewind();
            itemRead = this.channel.read(itemBuf);
            if (itemRead < 28L) {
                next = -1L;
            } else {
                itemBuf.flip();
                itemMsgPartId = itemBuf.getInt();
                itemMsgOffset = itemBuf.getLong();
                itemMsglen = itemBuf.getInt();
                itemKeyCode = itemBuf.getInt();
                itemTimeRecv = itemBuf.getLong();
                itemNext = validBytes + 28L;
                next = itemMsgPartId < 0 || itemMsgOffset < 0L || itemMsglen <= 0 || itemMsglen > 29360180 || itemNext > totalBytes ? -1L : itemNext;
            }
            if (next < 0L) continue;
            validBytes = next;
        } while (next >= 0L);
        if (totalBytes != validBytes) {
            this.channel.truncate(validBytes);
        }
        this.cachedSize.set(validBytes);
        this.flushedSize.set(validBytes);
        this.channel.position(validBytes);
        return new RecoverResult(totalBytes - validBytes, false);
    }

    private static class RecoverResult {
        private final long truncated;
        private final boolean isEqual;

        public RecoverResult(long truncated, boolean isEqual) {
            this.truncated = truncated;
            this.isEqual = isEqual;
        }

        public long getTruncated() {
            return this.truncated;
        }

        public boolean isEqual() {
            return this.isEqual;
        }
    }
}

