"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileSystemDataAdaptor = void 0;
var fs = _interopRequireWildcard(require("fs/promises"));
var _path = _interopRequireDefault(require("path"));
var _json = _interopRequireDefault(require("json5"));
var _utils = require("./utils");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
// Check if a location is a directory without an exception if location not found
const safeIsDirectory = async maybeDirectory => {
  try {
    return (await fs.lstat(maybeDirectory)).isDirectory();
  } catch (_err) {
    return false;
  }
};

/**
 * A CatalogDataAdaptor that reads from the local filesystem.
 * Used to read default Integrations shipped in the in-product catalog at `__data__`.
 */
class FileSystemDataAdaptor {
  /**
   * Creates a new FileSystemCatalogDataAdaptor instance.
   *
   * @param directory The base directory from which to read files. This is not sanitized.
   */
  constructor(directory) {
    _defineProperty(this, "isConfigLocalized", false);
    _defineProperty(this, "directory", void 0);
    this.directory = directory;
  }
  async readFile(filename, type) {
    let content;
    try {
      content = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename), {
        encoding: 'utf-8'
      });
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    // First try to parse as JSON, then NDJSON, then fail.
    try {
      const parsed = _json.default.parse(content);
      return {
        ok: true,
        value: parsed
      };
    } catch (err) {
      const parsed = await (0, _utils.tryParseNDJson)(content);
      if (parsed) {
        return {
          ok: true,
          value: parsed
        };
      }
      return {
        ok: false,
        error: new Error(`Unable to parse file '${filename}' as JSON or NDJson`, {
          cause: err
        })
      };
    }
  }
  async readFileRaw(filename, type) {
    try {
      const buffer = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename));
      return {
        ok: true,
        value: buffer
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async findIntegrations(dirname = '.') {
    try {
      const integrations = [];
      await this.collectIntegrationsRecursive(dirname, integrations);
      return {
        ok: true,
        value: integrations
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async collectIntegrationsRecursive(dirname, integrations) {
    const entries = await fs.readdir(_path.default.join(this.directory, dirname));
    for (const entry of entries) {
      const fullPath = _path.default.join(dirname, entry);
      const isDirectory = (await this.getDirectoryType(fullPath)) === 'integration';
      if (isDirectory) {
        integrations.push(fullPath);
      } else if ((await this.getDirectoryType(fullPath)) === 'repository') {
        await this.collectIntegrationsRecursive(fullPath, integrations);
      }
    }
  }
  async findIntegrationVersions(dirname = '.') {
    let files;
    const integPath = _path.default.join(this.directory, dirname);
    try {
      files = await fs.readdir(integPath);
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    const versions = [];
    for (const file of files) {
      // TODO handle nested repositories -- assumes integrations are 1 level deep
      if (_path.default.extname(file) === '.json' && file.startsWith(`${_path.default.basename(integPath)}-`)) {
        const version = file.substring(_path.default.basename(integPath).length + 1, file.length - 5);
        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }
        versions.push(version);
      }
    }
    return {
      ok: true,
      value: versions
    };
  }
  async getDirectoryType(dirname) {
    const isDir = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.'));
    if (!isDir) {
      return 'unknown';
    }
    // Sloppily just check for one mandatory integration directory to distinguish.
    // Improve if we need to distinguish a repository with an integration named "schemas".
    const hasSchemas = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.', 'schemas'));
    return hasSchemas ? 'integration' : 'repository';
  }
  join(filename) {
    return new FileSystemDataAdaptor(_path.default.join(this.directory, filename));
  }
}
exports.FileSystemDataAdaptor = FileSystemDataAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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