#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) 2021 Satpy developers
#
# This file is part of satpy.
#
# satpy is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# satpy is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# satpy.  If not, see <http://www.gnu.org/licenses/>.
"""Module for testing the satpy.readers.msi_safe module."""
import unittest.mock as mock
import warnings
from datetime import datetime
from io import BytesIO, StringIO

import dask.array as da
import numpy as np
import pytest
import xarray as xr

# Datetimes used for checking start time is correctly set.
fname_dt = datetime(2020, 10, 1, 18, 35, 41)
tilemd_dt = datetime(2020, 10, 1, 16, 34, 23, 153611)

mtd_l1c_tile_xml = b"""<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<n1:Level-1C_Tile_ID xmlns:n1="https://psd-14.sentinel2.eo.esa.int/PSD/S2_PDI_Level-1C_Tile_Metadata.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="https://psd-14.sentinel2.eo.esa.int/PSD/S2_PDI_Level-1C_Tile_Metadata.xsd /gpfs/dpc/app/s2ipf/FORMAT_METADATA_TILE_L1C/02.14.00/scripts/../../../schemas/02.17.00/PSD/S2_PDI_Level-1C_Tile_Metadata.xsd">

  <n1:General_Info>
    <TILE_ID metadataLevel="Brief">S2B_OPER_MSI_L1C_TL_VGS1_20201001T183541_A018656_T16SEB_N02.09</TILE_ID>
    <DATASTRIP_ID metadataLevel="Standard">S2B_OPER_MSI_L1C_DS_VGS1_20201001T183541_S20201001T162735_N02.09</DATASTRIP_ID>
    <DOWNLINK_PRIORITY metadataLevel="Standard">NOMINAL</DOWNLINK_PRIORITY>
    <SENSING_TIME metadataLevel="Standard">2020-10-01T16:34:23.153611Z</SENSING_TIME>
    <Archiving_Info metadataLevel="Expertise">
      <ARCHIVING_CENTRE>VGS1</ARCHIVING_CENTRE>
      <ARCHIVING_TIME>2020-10-01T18:55:55.59803Z</ARCHIVING_TIME>
    </Archiving_Info>
  </n1:General_Info>

  <n1:Geometric_Info>
    <Tile_Geocoding metadataLevel="Brief">
      <HORIZONTAL_CS_NAME>WGS84 / UTM zone 16N</HORIZONTAL_CS_NAME>
      <HORIZONTAL_CS_CODE>EPSG:32616</HORIZONTAL_CS_CODE>
      <Size resolution="10">
        <NROWS>10980</NROWS>
        <NCOLS>10980</NCOLS>
      </Size>
      <Size resolution="20">
        <NROWS>5490</NROWS>
        <NCOLS>5490</NCOLS>
      </Size>
      <Size resolution="60">
        <NROWS>1830</NROWS>
        <NCOLS>1830</NCOLS>
      </Size>
      <Geoposition resolution="10">
        <ULX>499980</ULX>
        <ULY>3700020</ULY>
        <XDIM>10</XDIM>
        <YDIM>-10</YDIM>
      </Geoposition>
      <Geoposition resolution="20">
        <ULX>499980</ULX>
        <ULY>3700020</ULY>
        <XDIM>20</XDIM>
        <YDIM>-20</YDIM>
      </Geoposition>
      <Geoposition resolution="60">
        <ULX>499980</ULX>
        <ULY>3700020</ULY>
        <XDIM>60</XDIM>
        <YDIM>-60</YDIM>
      </Geoposition>
    </Tile_Geocoding>
    <Tile_Angles metadataLevel="Standard">
      <Sun_Angles_Grid>
        <Zenith>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>39.8824 39.8636 39.8448 39.8261 39.8074 39.7888 39.7702 39.7516 39.7331 39.7145 39.6961 39.6776 39.6592 39.6408 39.6225 39.6042 39.5859 39.5677 39.5495 39.5313 39.5132 39.4951 39.477</VALUES>
            <VALUES>39.8404 39.8216 39.8029 39.7841 39.7655 39.7468 39.7282 39.7096 39.691 39.6725 39.654 39.6355 39.6171 39.5987 39.5804 39.5621 39.5438 39.5255 39.5073 39.4891 39.471 39.4529 39.4348</VALUES>
            <VALUES>39.7985 39.7797 39.7609 39.7422 39.7235 39.7048 39.6862 39.6675 39.649 39.6304 39.6119 39.5935 39.575 39.5566 39.5383 39.5199 39.5016 39.4834 39.4651 39.4469 39.4288 39.4107 39.3926</VALUES>
            <VALUES>39.7566 39.7377 39.719 39.7002 39.6815 39.6628 39.6441 39.6255 39.6069 39.5884 39.5699 39.5514 39.533 39.5145 39.4962 39.4778 39.4595 39.4412 39.423 39.4048 39.3866 39.3685 39.3504</VALUES>
            <VALUES>39.7146 39.6958 39.677 39.6582 39.6395 39.6208 39.6021 39.5835 39.5649 39.5464 39.5278 39.5093 39.4909 39.4724 39.4541 39.4357 39.4174 39.3991 39.3808 39.3626 39.3444 39.3263 39.3082</VALUES>
            <VALUES>39.6727 39.6539 39.635 39.6163 39.5975 39.5788 39.5601 39.5415 39.5229 39.5043 39.4858 39.4673 39.4488 39.4304 39.412 39.3936 39.3752 39.3569 39.3387 39.3204 39.3023 39.2841 39.266</VALUES>
            <VALUES>39.6308 39.6119 39.5931 39.5743 39.5556 39.5368 39.5181 39.4995 39.4809 39.4623 39.4437 39.4252 39.4067 39.3883 39.3699 39.3515 39.3331 39.3148 39.2965 39.2783 39.2601 39.2419 39.2238</VALUES>
            <VALUES>39.5889 39.57 39.5512 39.5324 39.5136 39.4949 39.4762 39.4575 39.4389 39.4203 39.4017 39.3832 39.3647 39.3462 39.3278 39.3094 39.291 39.2727 39.2544 39.2361 39.2179 39.1997 39.1816</VALUES>
            <VALUES>39.547 39.5281 39.5092 39.4904 39.4716 39.4529 39.4342 39.4155 39.3968 39.3782 39.3596 39.3411 39.3226 39.3041 39.2857 39.2673 39.2489 39.2306 39.2123 39.194 39.1758 39.1576 39.1394</VALUES>
            <VALUES>39.5051 39.4862 39.4673 39.4485 39.4297 39.4109 39.3922 39.3735 39.3548 39.3362 39.3176 39.2991 39.2805 39.2621 39.2436 39.2252 39.2068 39.1884 39.1701 39.1518 39.1336 39.1154 39.0972</VALUES>
            <VALUES>39.4632 39.4442 39.4254 39.4065 39.3877 39.3689 39.3502 39.3315 39.3128 39.2942 39.2756 39.257 39.2385 39.22 39.2015 39.1831 39.1647 39.1463 39.128 39.1097 39.0914 39.0732 39.055</VALUES>
            <VALUES>39.4213 39.4023 39.3834 39.3646 39.3458 39.327 39.3082 39.2895 39.2708 39.2522 39.2336 39.215 39.1964 39.1779 39.1594 39.141 39.1226 39.1042 39.0859 39.0676 39.0493 39.0311 39.0129</VALUES>
            <VALUES>39.3794 39.3604 39.3415 39.3227 39.3038 39.285 39.2663 39.2475 39.2288 39.2102 39.1915 39.1729 39.1544 39.1359 39.1174 39.0989 39.0805 39.0621 39.0438 39.0254 39.0072 38.9889 38.9707</VALUES>
            <VALUES>39.3375 39.3185 39.2996 39.2807 39.2619 39.2431 39.2243 39.2056 39.1868 39.1682 39.1495 39.1309 39.1123 39.0938 39.0753 39.0568 39.0384 39.02 39.0016 38.9833 38.965 38.9468 38.9285</VALUES>
            <VALUES>39.2956 39.2766 39.2577 39.2388 39.22 39.2011 39.1823 39.1636 39.1449 39.1262 39.1075 39.0889 39.0703 39.0518 39.0332 39.0148 38.9963 38.9779 38.9595 38.9412 38.9229 38.9046 38.8864</VALUES>
            <VALUES>39.2537 39.2348 39.2158 39.1969 39.178 39.1592 39.1404 39.1216 39.1029 39.0842 39.0655 39.0469 39.0283 39.0097 38.9912 38.9727 38.9542 38.9358 38.9174 38.8991 38.8807 38.8625 38.8442</VALUES>
            <VALUES>39.2119 39.1929 39.1739 39.155 39.1361 39.1173 39.0984 39.0797 39.0609 39.0422 39.0235 39.0049 38.9862 38.9677 38.9491 38.9306 38.9122 38.8937 38.8753 38.8569 38.8386 38.8203 38.8021</VALUES>
            <VALUES>39.17 39.151 39.132 39.1131 39.0942 39.0753 39.0565 39.0377 39.0189 39.0002 38.9815 38.9628 38.9442 38.9256 38.9071 38.8886 38.8701 38.8516 38.8332 38.8148 38.7965 38.7782 38.7599</VALUES>
            <VALUES>39.1281 39.1091 39.0901 39.0712 39.0523 39.0334 39.0145 38.9957 38.977 38.9582 38.9395 38.9208 38.9022 38.8836 38.865 38.8465 38.828 38.8095 38.7911 38.7727 38.7544 38.736 38.7178</VALUES>
            <VALUES>39.0863 39.0672 39.0482 39.0293 39.0104 38.9915 38.9726 38.9538 38.935 38.9162 38.8975 38.8788 38.8602 38.8416 38.823 38.8045 38.7859 38.7675 38.749 38.7306 38.7122 38.6939 38.6756</VALUES>
            <VALUES>39.0444 39.0254 39.0064 38.9874 38.9685 38.9496 38.9307 38.9118 38.893 38.8743 38.8555 38.8368 38.8182 38.7996 38.781 38.7624 38.7439 38.7254 38.7069 38.6885 38.6701 38.6518 38.6335</VALUES>
            <VALUES>39.0026 38.9835 38.9645 38.9455 38.9266 38.9076 38.8888 38.8699 38.8511 38.8323 38.8136 38.7949 38.7762 38.7575 38.7389 38.7204 38.7018 38.6833 38.6649 38.6464 38.628 38.6097 38.5913</VALUES>
            <VALUES>38.9607 38.9417 38.9226 38.9036 38.8847 38.8657 38.8468 38.828 38.8091 38.7903 38.7716 38.7529 38.7342 38.7155 38.6969 38.6783 38.6598 38.6413 38.6228 38.6043 38.5859 38.5676 38.5492</VALUES>
          </Values_List>
        </Zenith>
        <Azimuth>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>154.971 155.049 155.126 155.204 155.282 155.359 155.437 155.515 155.593 155.671 155.749 155.827 155.905 155.983 156.061 156.14 156.218 156.296 156.375 156.453 156.532 156.61 156.689</VALUES>
            <VALUES>154.953 155.03 155.108 155.186 155.263 155.341 155.419 155.497 155.575 155.653 155.731 155.809 155.887 155.965 156.043 156.122 156.2 156.278 156.357 156.435 156.514 156.592 156.671</VALUES>
            <VALUES>154.934 155.012 155.09 155.167 155.245 155.323 155.401 155.478 155.556 155.634 155.712 155.79 155.869 155.947 156.025 156.103 156.182 156.26 156.338 156.417 156.495 156.574 156.653</VALUES>
            <VALUES>154.916 154.994 155.071 155.149 155.227 155.304 155.382 155.46 155.538 155.616 155.694 155.772 155.85 155.928 156.007 156.085 156.163 156.242 156.32 156.399 156.477 156.556 156.634</VALUES>
            <VALUES>154.897 154.975 155.053 155.13 155.208 155.286 155.364 155.442 155.52 155.598 155.676 155.754 155.832 155.91 155.988 156.067 156.145 156.223 156.302 156.38 156.459 156.538 156.616</VALUES>
            <VALUES>154.879 154.956 155.034 155.112 155.19 155.267 155.345 155.423 155.501 155.579 155.657 155.735 155.814 155.892 155.97 156.048 156.127 156.205 156.284 156.362 156.441 156.519 156.598</VALUES>
            <VALUES>154.86 154.938 155.015 155.093 155.171 155.249 155.327 155.405 155.483 155.561 155.639 155.717 155.795 155.873 155.952 156.03 156.108 156.187 156.265 156.344 156.422 156.501 156.58</VALUES>
            <VALUES>154.841 154.919 154.997 155.075 155.152 155.23 155.308 155.386 155.464 155.542 155.62 155.698 155.777 155.855 155.933 156.012 156.09 156.168 156.247 156.325 156.404 156.483 156.561</VALUES>
            <VALUES>154.823 154.9 154.978 155.056 155.134 155.212 155.289 155.367 155.445 155.524 155.602 155.68 155.758 155.836 155.915 155.993 156.071 156.15 156.228 156.307 156.386 156.464 156.543</VALUES>
            <VALUES>154.804 154.882 154.959 155.037 155.115 155.193 155.271 155.349 155.427 155.505 155.583 155.661 155.739 155.818 155.896 155.974 156.053 156.131 156.21 156.289 156.367 156.446 156.525</VALUES>
            <VALUES>154.785 154.863 154.941 155.018 155.096 155.174 155.252 155.33 155.408 155.486 155.564 155.643 155.721 155.799 155.878 155.956 156.034 156.113 156.191 156.27 156.349 156.427 156.506</VALUES>
            <VALUES>154.766 154.844 154.922 155 155.077 155.155 155.233 155.311 155.389 155.467 155.546 155.624 155.702 155.78 155.859 155.937 156.016 156.094 156.173 156.251 156.33 156.409 156.488</VALUES>
            <VALUES>154.747 154.825 154.903 154.981 155.059 155.136 155.214 155.292 155.371 155.449 155.527 155.605 155.683 155.762 155.84 155.919 155.997 156.076 156.154 156.233 156.312 156.39 156.469</VALUES>
            <VALUES>154.728 154.806 154.884 154.962 155.04 155.118 155.196 155.274 155.352 155.43 155.508 155.586 155.665 155.743 155.821 155.9 155.978 156.057 156.136 156.214 156.293 156.372 156.451</VALUES>
            <VALUES>154.709 154.787 154.865 154.943 155.021 155.099 155.177 155.255 155.333 155.411 155.489 155.568 155.646 155.724 155.803 155.881 155.96 156.038 156.117 156.196 156.274 156.353 156.432</VALUES>
            <VALUES>154.69 154.768 154.846 154.924 155.002 155.08 155.158 155.236 155.314 155.392 155.47 155.549 155.627 155.705 155.784 155.862 155.941 156.019 156.098 156.177 156.256 156.334 156.413</VALUES>
            <VALUES>154.671 154.749 154.827 154.905 154.983 155.061 155.139 155.217 155.295 155.373 155.451 155.53 155.608 155.686 155.765 155.843 155.922 156.001 156.079 156.158 156.237 156.316 156.394</VALUES>
            <VALUES>154.652 154.73 154.808 154.886 154.964 155.042 155.12 155.198 155.276 155.354 155.432 155.511 155.589 155.668 155.746 155.825 155.903 155.982 156.06 156.139 156.218 156.297 156.376</VALUES>
            <VALUES>154.633 154.711 154.789 154.866 154.944 155.022 155.101 155.179 155.257 155.335 155.413 155.492 155.57 155.649 155.727 155.806 155.884 155.963 156.042 156.12 156.199 156.278 156.357</VALUES>
            <VALUES>154.614 154.691 154.769 154.847 154.925 155.003 155.081 155.16 155.238 155.316 155.394 155.473 155.551 155.63 155.708 155.787 155.865 155.944 156.023 156.101 156.18 156.259 156.338</VALUES>
            <VALUES>154.594 154.672 154.75 154.828 154.906 154.984 155.062 155.14 155.219 155.297 155.375 155.454 155.532 155.61 155.689 155.768 155.846 155.925 156.004 156.082 156.161 156.24 156.319</VALUES>
            <VALUES>154.575 154.653 154.731 154.809 154.887 154.965 155.043 155.121 155.199 155.278 155.356 155.434 155.513 155.591 155.67 155.748 155.827 155.906 155.985 156.063 156.142 156.221 156.3</VALUES>
            <VALUES>154.556 154.633 154.711 154.789 154.867 154.945 155.024 155.102 155.18 155.258 155.337 155.415 155.494 155.572 155.651 155.729 155.808 155.887 155.965 156.044 156.123 156.202 156.281</VALUES>
          </Values_List>
        </Azimuth>
      </Sun_Angles_Grid>
      <Mean_Sun_Angle>
        <ZENITH_ANGLE unit="deg">39.2158335161115</ZENITH_ANGLE>
        <AZIMUTH_ANGLE unit="deg">155.62398389104</AZIMUTH_ANGLE>
      </Mean_Sun_Angle>
      <Viewing_Incidence_Angles_Grids bandId="1" detectorId="1">
        <Zenith>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN 11.7128 11.3368 10.9601 10.5837 10.2053 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN 11.6285 11.2531 10.8763 10.4977 10.1207 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>11.9203 11.5439 11.1676 10.79 10.4135 10.036 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>11.8359 11.4595 11.0825 10.7054 10.3284 9.95143 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>11.751 11.3743 10.9977 10.6209 10.2437 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>11.6664 11.2901 10.9134 10.5362 10.1591 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>11.5818 11.2061 10.8293 10.4518 10.0747 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>11.4976 11.121 10.7439 10.3664 9.98937 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Zenith>
        <Azimuth>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN 111.269 111.67 112.096 112.551 113.041 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN 111.354 111.759 112.192 112.657 113.152 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>111.053 111.44 111.852 112.292 112.762 113.266 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>111.136 111.529 111.946 112.392 112.869 113.381 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>111.219 111.618 112.042 112.494 112.978 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>111.305 111.708 112.138 112.597 113.089 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>111.391 111.799 112.235 112.702 113.201 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>111.478 111.893 112.336 112.809 113.317 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Azimuth>
      </Viewing_Incidence_Angles_Grids>
      <Viewing_Incidence_Angles_Grids bandId="1" detectorId="2">
        <Zenith>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN 9.82039 9.4373 9.05284 8.66805 8.28339 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN 9.73454 9.35159 8.96724 8.58182 8.19763 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 10.0315 9.64827 9.26401 8.87996 8.49572 8.11079 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 9.94572 9.56205 9.17796 8.79367 8.4095 8.02451 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 9.85977 9.47669 9.09189 8.70763 8.32282 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 9.77437 9.38968 9.00597 8.62183 8.23655 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 9.68751 9.30377 8.91958 8.53514 8.15057 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN 9.98449 9.60143 9.21746 8.83286 8.4486 8.06421 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 9.51494 9.13074 8.74664 8.3621 7.97741 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Zenith>
        <Azimuth>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN 92.2969 91.9939 91.6606 91.294 90.8911 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN 92.2267 91.9172 91.5775 91.2031 90.7918 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 92.4452 92.1553 91.8379 91.4911 91.1101 90.6885 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 92.3781 92.0825 91.7591 91.4043 91.0144 90.5834 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 92.31 92.0089 91.6783 91.3163 90.9166 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 92.2413 91.9324 91.5954 91.2255 90.8166 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 92.1696 91.8556 91.5111 91.1322 90.7147 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN 92.391 92.0976 91.7769 91.4248 91.0382 90.611 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN 92.0248 91.6966 91.3373 90.9417 90.5043 NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Azimuth>
      </Viewing_Incidence_Angles_Grids>
      <Viewing_Incidence_Angles_Grids bandId="1" detectorId="3">
        <Zenith>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.94194 7.56511 7.19038 6.81626 6.44423 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.8569 7.48093 7.10605 6.73281 6.36089 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.7724 7.39658 7.02215 6.64892 6.27782 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 8.06455 7.688 7.31247 6.93823 6.56551 6.19477 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.97983 7.60366 7.2287 6.85441 6.48197 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.8952 7.51946 7.14517 6.77071 6.39873 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.81105 7.43489 7.0603 6.68714 6.31558 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.72611 7.35074 6.97674 6.60389 6.23289 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN 8.01804 7.64172 7.26672 6.89282 6.52025 6.14959 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 7.55748 7.18239 6.80886 6.43657 NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Zenith>
        <Azimuth>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 116.798 117.613 118.509 119.504 120.609 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 116.97 117.802 118.719 119.735 120.87 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 117.146 117.996 118.934 119.975 121.137 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 116.538 117.326 118.194 119.155 120.222 121.414 NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 116.706 117.511 118.397 119.38 120.474 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 116.877 117.699 118.604 119.612 120.733 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 117.05 117.892 118.82 119.85 120.998 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 117.231 118.089 119.037 120.092 121.27 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN 116.616 117.414 118.291 119.262 120.343 121.552 NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 117.601 118.499 119.492 120.6 NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Azimuth>
      </Viewing_Incidence_Angles_Grids>
      <Viewing_Incidence_Angles_Grids bandId="1" detectorId="4">
        <Zenith>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.98397 5.60436 5.22629 4.85051 4.47749 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.89902 5.51979 5.14214 4.76699 4.39482 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.81385 5.43495 5.05811 4.68338 4.31176 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 6.10878 5.72912 5.35071 4.97413 4.59998 4.22933 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 6.02356 5.64376 5.26618 4.88984 4.51664 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.938 5.55897 5.1813 4.80571 4.43316 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.85332 5.47505 5.09703 4.72192 4.35017 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 6.14847 5.76823 5.38949 5.01237 4.63811 4.26692 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 6.06275 5.68347 5.30458 4.92804 4.55459 4.18407 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.9774 5.59788 5.21981 4.84402 4.47086 NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 5.51271 5.13498 4.7597 4.38749 NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Zenith>
        <Azimuth>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 87.1015 86.1123 84.974 83.6538 82.1077 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 86.8857 85.8663 84.6903 83.3238 81.7192 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 86.6628 85.6105 84.3968 82.9801 81.3118 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 87.3809 86.4344 85.3483 84.0925 82.6251 80.8924 NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 87.1741 86.1977 85.0768 83.7748 82.2541 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 86.9589 85.9527 84.7944 83.4481 81.8691 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 86.7408 85.7045 84.5049 83.1103 81.4708 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 87.4494 86.5139 85.442 84.2014 82.7561 81.052 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 87.244 86.2812 85.1729 83.8909 82.3929 80.6189 NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 87.0343 86.0398 84.8955 83.5691 82.0132 NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 85.7916 84.6089 83.2341 81.6205 NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Azimuth>
      </Viewing_Incidence_Angles_Grids>
      <Viewing_Incidence_Angles_Grids bandId="1" detectorId="5">
        <Zenith>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.27277 3.93031</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.19493 3.85385</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.11765 3.77876</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.38681 4.04091 3.70407</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.30823 3.96401 3.63007</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.22988 3.88788 3.55663</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.15225 3.8125 3.48381</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.4219 4.07481 3.73746 3.41201</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.34311 3.998 3.66286 3.34095</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.26482 3.92174 3.58929 3.27063</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 4.18686 3.84597 3.51627 3.20131</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 3.7708 3.44395 3.13291</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Zenith>
        <Azimuth>
          <COL_STEP unit="m">5000</COL_STEP>
          <ROW_STEP unit="m">5000</ROW_STEP>
          <Values_List>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 130.971 133.734</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 131.547 134.423</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 132.144 135.129</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 130.141 132.763 135.869</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 130.69 133.411 136.637</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 131.26 134.084 137.44</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 131.852 134.784 138.279</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 129.884 132.465 135.512 139.15</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 130.424 133.101 136.272 140.06</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 130.987 133.764 137.059 141.008</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 131.568 134.453 137.883 142.001</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN 135.169 138.743 143.037</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
            <VALUES>NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN</VALUES>
          </Values_List>
        </Azimuth>
      </Viewing_Incidence_Angles_Grids>
      <Mean_Viewing_Incidence_Angle_List>
        <Mean_Viewing_Incidence_Angle bandId="0">
          <ZENITH_ANGLE unit="deg">7.18024135920399</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">106.255157702848</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="9">
          <ZENITH_ANGLE unit="deg">7.22336464325122</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">106.346078097961</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="10">
          <ZENITH_ANGLE unit="deg">6.98729381785528</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.765382381968</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="1">
          <ZENITH_ANGLE unit="deg">6.92446640065506</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.09376719949</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="2">
          <ZENITH_ANGLE unit="deg">6.95791117837005</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.44993173891</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="3">
          <ZENITH_ANGLE unit="deg">6.99577507894955</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.692478311655</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="4">
          <ZENITH_ANGLE unit="deg">7.0231940118902</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.844309043016</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="5">
          <ZENITH_ANGLE unit="deg">7.05361232723533</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.969629461909</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="6">
          <ZENITH_ANGLE unit="deg">7.0871211819946</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">106.101277617057</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="7">
          <ZENITH_ANGLE unit="deg">6.93953882104395</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">105.275888180279</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="8">
          <ZENITH_ANGLE unit="deg">7.12343057570894</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">106.21247453177</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="11">
          <ZENITH_ANGLE unit="deg">7.04938612963508</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">106.030547019406</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
        <Mean_Viewing_Incidence_Angle bandId="12">
          <ZENITH_ANGLE unit="deg">7.13282515906901</ZENITH_ANGLE>
          <AZIMUTH_ANGLE unit="deg">106.31610702063</AZIMUTH_ANGLE>
        </Mean_Viewing_Incidence_Angle>
      </Mean_Viewing_Incidence_Angle_List>
    </Tile_Angles>
  </n1:Geometric_Info>

  <n1:Quality_Indicators_Info metadataLevel="Standard">
    <Image_Content_QI>
      <CLOUDY_PIXEL_PERCENTAGE>0</CLOUDY_PIXEL_PERCENTAGE>
      <DEGRADED_MSI_DATA_PERCENTAGE>0</DEGRADED_MSI_DATA_PERCENTAGE>
    </Image_Content_QI>
    <Pixel_Level_QI geometry="FULL_RESOLUTION">
      <MASK_FILENAME bandId="0" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B01.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="0" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B01.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="0" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B01.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="0" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B01.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="0" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B01.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="1" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B02.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="1" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B02.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="1" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B02.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="1" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B02.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="1" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B02.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="2" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B03.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="2" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B03.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="2" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B03.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="2" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B03.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="2" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B03.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="3" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B04.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="3" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B04.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="3" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B04.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="3" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B04.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="3" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B04.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="4" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B05.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="4" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B05.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="4" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B05.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="4" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B05.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="4" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B05.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="5" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B06.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="5" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B06.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="5" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B06.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="5" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B06.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="5" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B06.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="6" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B07.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="6" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B07.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="6" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B07.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="6" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B07.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="6" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B07.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="7" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B08.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="7" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B08.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="7" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B08.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="7" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B08.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="7" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B08.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="8" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B8A.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="8" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B8A.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="8" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B8A.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="8" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B8A.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="8" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B8A.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="9" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B09.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="9" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B09.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="9" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B09.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="9" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B09.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="9" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B09.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="10" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B10.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="10" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B10.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="10" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B10.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="10" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B10.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="10" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B10.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="11" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B11.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="11" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B11.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="11" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B11.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="11" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B11.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="11" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B11.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="12" type="MSK_DEFECT">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DEFECT_B12.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="12" type="MSK_DETFOO">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_DETFOO_B12.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="12" type="MSK_NODATA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_NODATA_B12.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="12" type="MSK_SATURA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_SATURA_B12.gml</MASK_FILENAME>
      <MASK_FILENAME bandId="12" type="MSK_TECQUA">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_TECQUA_B12.gml</MASK_FILENAME>
      <MASK_FILENAME type="MSK_CLOUDS">GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/MSK_CLOUDS_B00.gml</MASK_FILENAME>
    </Pixel_Level_QI>
    <PVI_FILENAME>GRANULE/L1C_T16SEB_A018656_20201001T162735/QI_DATA/T16SEB_20201001T162019_PVI.jp2</PVI_FILENAME>
  </n1:Quality_Indicators_Info>

</n1:Level-1C_Tile_ID>
"""  # noqa

mtd_l1c_old_xml = """<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<n1:Level-1C_User_Product xmlns:n1="https://psd-14.sentinel2.eo.esa.int/PSD/User_Product_Level-1C.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="https://psd-14.sentinel2.eo.esa.int/PSD/User_Product_Level-1C.xsd">
  <n1:General_Info>
    <Product_Info>
      <PRODUCT_START_TIME>2021-05-17T10:36:19.024Z</PRODUCT_START_TIME>
      <PRODUCT_STOP_TIME>2021-05-17T10:36:19.024Z</PRODUCT_STOP_TIME>
      <PRODUCT_URI>S2B_MSIL1C_20210517T103619_N7990_R008_T30QVE_20210929T075738.SAFE</PRODUCT_URI>
      <PROCESSING_LEVEL>Level-1C</PROCESSING_LEVEL>
      <PRODUCT_TYPE>S2MSI1C</PRODUCT_TYPE>
      <PROCESSING_BASELINE>79.90</PROCESSING_BASELINE>
      <PRODUCT_DOI>https://doi.org/10.5270/S2_-742ikth</PRODUCT_DOI>
      <GENERATION_TIME>2021-09-29T07:57:38.000000Z</GENERATION_TIME>
      <PREVIEW_IMAGE_URL>Not applicable</PREVIEW_IMAGE_URL>
      <PREVIEW_GEO_INFO>Not applicable</PREVIEW_GEO_INFO>
      <Datatake datatakeIdentifier="GS2B_20210517T103619_021913_N79.90">
        <SPACECRAFT_NAME>Sentinel-2B</SPACECRAFT_NAME>
        <DATATAKE_TYPE>INS-NOBS</DATATAKE_TYPE>
        <DATATAKE_SENSING_START>2021-05-17T10:36:19.024Z</DATATAKE_SENSING_START>
        <SENSING_ORBIT_NUMBER>8</SENSING_ORBIT_NUMBER>
        <SENSING_ORBIT_DIRECTION>DESCENDING</SENSING_ORBIT_DIRECTION>
      </Datatake>
      <Query_Options completeSingleTile="true">
        <PRODUCT_FORMAT>SAFE_COMPACT</PRODUCT_FORMAT>
      </Query_Options>
      <Product_Organisation>
        <Granule_List>
          <Granule datastripIdentifier="S2B_OPER_MSI_L1C_DS_VGSR_20210929T075738_S20210517T104617_N79.90" granuleIdentifier="S2B_OPER_MSI_L1C_TL_VGSR_20210929T075738_A021913_T30QVE_N79.90" imageFormat="JPEG2000">
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B01</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B02</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B03</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B04</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B05</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B06</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B07</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B08</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B8A</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B09</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B10</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B11</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B12</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_TCI</IMAGE_FILE>
          </Granule>
        </Granule_List>
      </Product_Organisation>
    </Product_Info>
    <Product_Image_Characteristics>
      <Special_Values>
        <SPECIAL_VALUE_TEXT>NODATA</SPECIAL_VALUE_TEXT>
        <SPECIAL_VALUE_INDEX>0</SPECIAL_VALUE_INDEX>
      </Special_Values>
      <Special_Values>
        <SPECIAL_VALUE_TEXT>SATURATED</SPECIAL_VALUE_TEXT>
        <SPECIAL_VALUE_INDEX>65535</SPECIAL_VALUE_INDEX>
      </Special_Values>
      <Image_Display_Order>
        <RED_CHANNEL>3</RED_CHANNEL>
        <GREEN_CHANNEL>2</GREEN_CHANNEL>
        <BLUE_CHANNEL>1</BLUE_CHANNEL>
      </Image_Display_Order>
      <QUANTIFICATION_VALUE unit="none">10000</QUANTIFICATION_VALUE>
      <Reflectance_Conversion>
        <U>0.979428313059035</U>
        <Solar_Irradiance_List>
          <SOLAR_IRRADIANCE bandId="0" unit="W/m²/µm">1874.3</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="1" unit="W/m²/µm">1959.75</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="2" unit="W/m²/µm">1824.93</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="3" unit="W/m²/µm">1512.79</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="4" unit="W/m²/µm">1425.78</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="5" unit="W/m²/µm">1291.13</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="6" unit="W/m²/µm">1175.57</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="7" unit="W/m²/µm">1041.28</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="8" unit="W/m²/µm">953.93</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="9" unit="W/m²/µm">817.58</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="10" unit="W/m²/µm">365.41</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="11" unit="W/m²/µm">247.08</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="12" unit="W/m²/µm">87.75</SOLAR_IRRADIANCE>
        </Solar_Irradiance_List>
      </Reflectance_Conversion>
      <Spectral_Information_List>
        <Spectral_Information bandId="0" physicalBand="B1">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">411</MIN>
            <MAX unit="nm">456</MAX>
            <CENTRAL unit="nm">442.3</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.0062411 0.01024045 0.00402983 0.00642179 0.00552753 0.0065525 0.00409887 0.006297 0.00436742 0.00233356 0.00058162 0.00202276 0.00294328 0.00485362 0.00317041 0.00237657 0.00234612 0.00440152 0.01292397 0.05001678 0.18650104 0.45441623 0.72307877 0.83999211 0.86456334 0.87472096 0.89215296 0.91090814 0.92588017 0.93924094 0.94491826 0.95078529 0.96803023 0.99939195 1 0.97548364 0.96148351 0.94986211 0.91841452 0.87989802 0.80383677 0.59752075 0.30474132 0.10798014 0.0304465 0.00885119</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="1" physicalBand="B2">
          <RESOLUTION>10</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">456</MIN>
            <MAX unit="nm">532</MAX>
            <CENTRAL unit="nm">492.3</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.05529541 0.12005068 0.25199051 0.4623617 0.65162379 0.77642171 0.82319091 0.83083116 0.83382106 0.837526 0.86304286 0.88226141 0.90486326 0.92043837 0.93602675 0.930533 0.92714067 0.9161479 0.90551724 0.89745515 0.90266694 0.90854264 0.92047913 0.92417935 0.91845025 0.90743244 0.89733983 0.88646415 0.87189983 0.85643973 0.84473414 0.84190734 0.85644111 0.87782724 0.90261174 0.91840544 0.94585847 0.96887192 0.99336135 0.99927899 1 0.99520325 0.98412711 0.97947473 0.97808297 0.97213439 0.96277794 0.95342234 0.93802376 0.92460144 0.90932642 0.90192251 0.89184298 0.88963556 0.89146958 0.89877911 0.91056869 0.92427362 0.93823555 0.95311791 0.97150808 0.98737003 0.99658514 0.99367959 0.98144714 0.95874415 0.89291635 0.73566218 0.52060373 0.3322804 0.19492197 0.11732617 0.07507304 0.05094154 0.03213016 0.01510217 0.00447984</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="10" physicalBand="B10">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">1339</MIN>
            <MAX unit="nm">1415</MAX>
            <CENTRAL unit="nm">1376.9</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>2.472e-05 0.00013691 0.00012558 8.901e-05 0.00012425 9.941e-05 0.00013952 0.00015816 0.00019272 0.00025959 0.00032221 0.00034719 0.0003699 0.00054874 0.00105434 0.00218813 0.00480743 0.01135252 0.02671185 0.05776022 0.11176337 0.19587518 0.31418191 0.46188068 0.62292578 0.7709851 0.88086652 0.9448941 0.97405066 0.98616696 0.99306955 0.99775441 1 0.99942348 0.99616891 0.99082045 0.9842131 0.97708513 0.97013647 0.96374366 0.95755001 0.95127438 0.94546638 0.94069659 0.93759595 0.93624612 0.93510206 0.93054472 0.91630845 0.88530334 0.83129653 0.74856466 0.63524397 0.49733159 0.34907723 0.21259735 0.10971453 0.04789269 0.01853013 0.00716776 0.0031533 0.00157017 0.00084901 0.00053006 0.00033171 0.00019447 0.00022104 0.00022646 0.00018156 0.00016063 0.00015475 0.00014734 0.00014776 0.00017405 0.00023619 0.00012007 4.337e-05</VALUES>
          </Spectral_Response>
        </Spectral_Information>
      </Spectral_Information_List>
      <PHYSICAL_GAINS bandId="0">3.97083657</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="1">3.81081866</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="2">4.21881648</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="3">4.7545091</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="4">5.16489535</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="5">5.06418355</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="6">4.7429031</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="7">6.789537</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="8">5.73223234</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="9">9.32447797</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="10">56.36387909</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="11">37.15464608</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="12">108.67071783</PHYSICAL_GAINS>
      <REFERENCE_BAND>3</REFERENCE_BAND>
    </Product_Image_Characteristics>
  </n1:General_Info>
</n1:Level-1C_User_Product>
"""  # noqa

mtd_l1c_xml = """<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<n1:Level-1C_User_Product xmlns:n1="https://psd-14.sentinel2.eo.esa.int/PSD/User_Product_Level-1C.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="https://psd-14.sentinel2.eo.esa.int/PSD/User_Product_Level-1C.xsd">
  <n1:General_Info>
    <Product_Info>
      <PRODUCT_START_TIME>2021-05-17T10:36:19.024Z</PRODUCT_START_TIME>
      <PRODUCT_STOP_TIME>2021-05-17T10:36:19.024Z</PRODUCT_STOP_TIME>
      <PRODUCT_URI>S2B_MSIL1C_20210517T103619_N7990_R008_T30QVE_20210929T075738.SAFE</PRODUCT_URI>
      <PROCESSING_LEVEL>Level-1C</PROCESSING_LEVEL>
      <PRODUCT_TYPE>S2MSI1C</PRODUCT_TYPE>
      <PROCESSING_BASELINE>79.90</PROCESSING_BASELINE>
      <PRODUCT_DOI>https://doi.org/10.5270/S2_-742ikth</PRODUCT_DOI>
      <GENERATION_TIME>2021-09-29T07:57:38.000000Z</GENERATION_TIME>
      <PREVIEW_IMAGE_URL>Not applicable</PREVIEW_IMAGE_URL>
      <PREVIEW_GEO_INFO>Not applicable</PREVIEW_GEO_INFO>
      <Datatake datatakeIdentifier="GS2B_20210517T103619_021913_N79.90">
        <SPACECRAFT_NAME>Sentinel-2B</SPACECRAFT_NAME>
        <DATATAKE_TYPE>INS-NOBS</DATATAKE_TYPE>
        <DATATAKE_SENSING_START>2021-05-17T10:36:19.024Z</DATATAKE_SENSING_START>
        <SENSING_ORBIT_NUMBER>8</SENSING_ORBIT_NUMBER>
        <SENSING_ORBIT_DIRECTION>DESCENDING</SENSING_ORBIT_DIRECTION>
      </Datatake>
      <Query_Options completeSingleTile="true">
        <PRODUCT_FORMAT>SAFE_COMPACT</PRODUCT_FORMAT>
      </Query_Options>
      <Product_Organisation>
        <Granule_List>
          <Granule datastripIdentifier="S2B_OPER_MSI_L1C_DS_VGSR_20210929T075738_S20210517T104617_N79.90" granuleIdentifier="S2B_OPER_MSI_L1C_TL_VGSR_20210929T075738_A021913_T30QVE_N79.90" imageFormat="JPEG2000">
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B01</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B02</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B03</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B04</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B05</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B06</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B07</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B08</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B8A</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B09</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B10</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B11</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_B12</IMAGE_FILE>
            <IMAGE_FILE>GRANULE/L1C_T30QVE_A021913_20210517T104617/IMG_DATA/T30QVE_20210517T103619_TCI</IMAGE_FILE>
          </Granule>
        </Granule_List>
      </Product_Organisation>
    </Product_Info>
    <Product_Image_Characteristics>
      <Special_Values>
        <SPECIAL_VALUE_TEXT>NODATA</SPECIAL_VALUE_TEXT>
        <SPECIAL_VALUE_INDEX>0</SPECIAL_VALUE_INDEX>
      </Special_Values>
      <Special_Values>
        <SPECIAL_VALUE_TEXT>SATURATED</SPECIAL_VALUE_TEXT>
        <SPECIAL_VALUE_INDEX>65535</SPECIAL_VALUE_INDEX>
      </Special_Values>
      <Image_Display_Order>
        <RED_CHANNEL>3</RED_CHANNEL>
        <GREEN_CHANNEL>2</GREEN_CHANNEL>
        <BLUE_CHANNEL>1</BLUE_CHANNEL>
      </Image_Display_Order>
      <QUANTIFICATION_VALUE unit="none">10000</QUANTIFICATION_VALUE>
      <Radiometric_Offset_List>
        <RADIO_ADD_OFFSET band_id="0">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="1">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="2">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="3">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="4">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="5">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="6">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="7">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="8">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="9">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="10">-2000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="11">-1000</RADIO_ADD_OFFSET>
        <RADIO_ADD_OFFSET band_id="12">-1000</RADIO_ADD_OFFSET>
      </Radiometric_Offset_List>
      <Reflectance_Conversion>
        <U>0.979428313059035</U>
        <Solar_Irradiance_List>
          <SOLAR_IRRADIANCE bandId="0" unit="W/m²/µm">1874.3</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="1" unit="W/m²/µm">1959.75</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="2" unit="W/m²/µm">1824.93</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="3" unit="W/m²/µm">1512.79</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="4" unit="W/m²/µm">1425.78</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="5" unit="W/m²/µm">1291.13</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="6" unit="W/m²/µm">1175.57</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="7" unit="W/m²/µm">1041.28</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="8" unit="W/m²/µm">953.93</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="9" unit="W/m²/µm">817.58</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="10" unit="W/m²/µm">365.41</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="11" unit="W/m²/µm">247.08</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="12" unit="W/m²/µm">87.75</SOLAR_IRRADIANCE>
        </Solar_Irradiance_List>
      </Reflectance_Conversion>
      <Spectral_Information_List>
        <Spectral_Information bandId="0" physicalBand="B1">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">411</MIN>
            <MAX unit="nm">456</MAX>
            <CENTRAL unit="nm">442.3</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.0062411 0.01024045 0.00402983 0.00642179 0.00552753 0.0065525 0.00409887 0.006297 0.00436742 0.00233356 0.00058162 0.00202276 0.00294328 0.00485362 0.00317041 0.00237657 0.00234612 0.00440152 0.01292397 0.05001678 0.18650104 0.45441623 0.72307877 0.83999211 0.86456334 0.87472096 0.89215296 0.91090814 0.92588017 0.93924094 0.94491826 0.95078529 0.96803023 0.99939195 1 0.97548364 0.96148351 0.94986211 0.91841452 0.87989802 0.80383677 0.59752075 0.30474132 0.10798014 0.0304465 0.00885119</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="1" physicalBand="B2">
          <RESOLUTION>10</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">456</MIN>
            <MAX unit="nm">532</MAX>
            <CENTRAL unit="nm">492.3</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.05529541 0.12005068 0.25199051 0.4623617 0.65162379 0.77642171 0.82319091 0.83083116 0.83382106 0.837526 0.86304286 0.88226141 0.90486326 0.92043837 0.93602675 0.930533 0.92714067 0.9161479 0.90551724 0.89745515 0.90266694 0.90854264 0.92047913 0.92417935 0.91845025 0.90743244 0.89733983 0.88646415 0.87189983 0.85643973 0.84473414 0.84190734 0.85644111 0.87782724 0.90261174 0.91840544 0.94585847 0.96887192 0.99336135 0.99927899 1 0.99520325 0.98412711 0.97947473 0.97808297 0.97213439 0.96277794 0.95342234 0.93802376 0.92460144 0.90932642 0.90192251 0.89184298 0.88963556 0.89146958 0.89877911 0.91056869 0.92427362 0.93823555 0.95311791 0.97150808 0.98737003 0.99658514 0.99367959 0.98144714 0.95874415 0.89291635 0.73566218 0.52060373 0.3322804 0.19492197 0.11732617 0.07507304 0.05094154 0.03213016 0.01510217 0.00447984</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="10" physicalBand="B10">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">1339</MIN>
            <MAX unit="nm">1415</MAX>
            <CENTRAL unit="nm">1376.9</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>2.472e-05 0.00013691 0.00012558 8.901e-05 0.00012425 9.941e-05 0.00013952 0.00015816 0.00019272 0.00025959 0.00032221 0.00034719 0.0003699 0.00054874 0.00105434 0.00218813 0.00480743 0.01135252 0.02671185 0.05776022 0.11176337 0.19587518 0.31418191 0.46188068 0.62292578 0.7709851 0.88086652 0.9448941 0.97405066 0.98616696 0.99306955 0.99775441 1 0.99942348 0.99616891 0.99082045 0.9842131 0.97708513 0.97013647 0.96374366 0.95755001 0.95127438 0.94546638 0.94069659 0.93759595 0.93624612 0.93510206 0.93054472 0.91630845 0.88530334 0.83129653 0.74856466 0.63524397 0.49733159 0.34907723 0.21259735 0.10971453 0.04789269 0.01853013 0.00716776 0.0031533 0.00157017 0.00084901 0.00053006 0.00033171 0.00019447 0.00022104 0.00022646 0.00018156 0.00016063 0.00015475 0.00014734 0.00014776 0.00017405 0.00023619 0.00012007 4.337e-05</VALUES>
          </Spectral_Response>
        </Spectral_Information>
      </Spectral_Information_List>
      <PHYSICAL_GAINS bandId="0">3.97083657</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="1">3.81081866</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="2">4.21881648</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="3">4.7545091</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="4">5.16489535</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="5">5.06418355</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="6">4.7429031</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="7">6.789537</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="8">5.73223234</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="9">9.32447797</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="10">56.36387909</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="11">37.15464608</PHYSICAL_GAINS>
      <PHYSICAL_GAINS bandId="12">108.67071783</PHYSICAL_GAINS>
      <REFERENCE_BAND>3</REFERENCE_BAND>
    </Product_Image_Characteristics>
  </n1:General_Info>
</n1:Level-1C_User_Product>
"""  # noqa

mtd_l2a_xml = """<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<n1:Level-2A_User_Product xmlns:n1="https://psd-14.sentinel2.eo.esa.int/PSD/User_Product_Level-2A.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="https://psd-14.sentinel2.eo.esa.int/PSD/User_Product_Level-2A.xsd">
    <n1:General_Info>
        <Product_Info>
            <PRODUCT_START_TIME>2024-04-11T03:05:21.024Z</PRODUCT_START_TIME>
            <PRODUCT_STOP_TIME>2024-04-11T03:05:21.024Z</PRODUCT_STOP_TIME>
            <PRODUCT_URI>S2A_MSIL2A_20240411T030521_N0510_R075_T50TMK_20240411T080950.SAFE</PRODUCT_URI>
            <PROCESSING_LEVEL>Level-2A</PROCESSING_LEVEL>
            <PRODUCT_TYPE>S2MSI2A</PRODUCT_TYPE>
            <PROCESSING_BASELINE>05.10</PROCESSING_BASELINE>
            <PRODUCT_DOI>https://doi.org/10.5270/S2_-znk9xsj</PRODUCT_DOI>
            <GENERATION_TIME>2024-04-11T08:09:50.000000Z</GENERATION_TIME>
            <PREVIEW_IMAGE_URL>Not applicable</PREVIEW_IMAGE_URL>
            <PREVIEW_GEO_INFO>Not applicable</PREVIEW_GEO_INFO>
            <Datatake datatakeIdentifier="GS2A_20240411T030521_045975_N05.10">
      <SPACECRAFT_NAME>Sentinel-2A</SPACECRAFT_NAME>
      <DATATAKE_TYPE>INS-NOBS</DATATAKE_TYPE>
      <DATATAKE_SENSING_START>2024-04-11T03:05:21.024Z</DATATAKE_SENSING_START>
      <SENSING_ORBIT_NUMBER>75</SENSING_ORBIT_NUMBER>
      <SENSING_ORBIT_DIRECTION>DESCENDING</SENSING_ORBIT_DIRECTION>
    </Datatake>
<Query_Options completeSingleTile="true">
<PRODUCT_FORMAT>SAFE_COMPACT</PRODUCT_FORMAT>
</Query_Options>
<Product_Organisation>
                <Granule_List>
                    <Granule datastripIdentifier="S2A_OPER_MSI_L2A_DS_2APS_20240411T080950_S20240411T030632_N05.10" granuleIdentifier="S2A_OPER_MSI_L2A_TL_2APS_20240411T080950_A045975_T50TMK_N05.10" imageFormat="JPEG2000">
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_B02_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_B03_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_B04_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_B08_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_TCI_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_AOT_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R10m/T50TMK_20240411T030521_WVP_10m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B01_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B02_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B03_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B04_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B05_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B06_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B07_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B8A_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B11_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_B12_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_TCI_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_AOT_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_WVP_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R20m/T50TMK_20240411T030521_SCL_20m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B01_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B02_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B03_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B04_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B05_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B06_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B07_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B8A_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B09_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B11_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_B12_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_TCI_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_AOT_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_WVP_60m</IMAGE_FILE>
                        <IMAGE_FILE>GRANULE/L2A_T50TMK_A045975_20240411T030632/IMG_DATA/R60m/T50TMK_20240411T030521_SCL_60m</IMAGE_FILE>
                    </Granule>
                </Granule_List>
</Product_Organisation>
        </Product_Info>
        <Product_Image_Characteristics>
            <Special_Values>
                <SPECIAL_VALUE_TEXT>NODATA</SPECIAL_VALUE_TEXT>
                <SPECIAL_VALUE_INDEX>0</SPECIAL_VALUE_INDEX>
            </Special_Values>
            <Special_Values>
                <SPECIAL_VALUE_TEXT>SATURATED</SPECIAL_VALUE_TEXT>
                <SPECIAL_VALUE_INDEX>65535</SPECIAL_VALUE_INDEX>
            </Special_Values>
            <Image_Display_Order>
                <RED_CHANNEL>3</RED_CHANNEL>
                <GREEN_CHANNEL>2</GREEN_CHANNEL>
                <BLUE_CHANNEL>1</BLUE_CHANNEL>
            </Image_Display_Order>
            <QUANTIFICATION_VALUES_LIST>
                <BOA_QUANTIFICATION_VALUE unit="none">10000</BOA_QUANTIFICATION_VALUE>
                <AOT_QUANTIFICATION_VALUE unit="none">1000.0</AOT_QUANTIFICATION_VALUE>
                <WVP_QUANTIFICATION_VALUE unit="cm">1000.0</WVP_QUANTIFICATION_VALUE>
            </QUANTIFICATION_VALUES_LIST>
            <BOA_ADD_OFFSET_VALUES_LIST>
        <BOA_ADD_OFFSET band_id="0">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="1">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="2">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="3">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="4">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="5">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="6">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="7">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="8">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="9">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="10">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="11">-1000</BOA_ADD_OFFSET>
        <BOA_ADD_OFFSET band_id="12">-1000</BOA_ADD_OFFSET>
      </BOA_ADD_OFFSET_VALUES_LIST>
            <Reflectance_Conversion>
        <U>0.998279632507911</U>
        <Solar_Irradiance_List>
          <SOLAR_IRRADIANCE bandId="0" unit="W/m²/µm">1884.69</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="1" unit="W/m²/µm">1959.66</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="2" unit="W/m²/µm">1823.24</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="3" unit="W/m²/µm">1512.06</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="4" unit="W/m²/µm">1424.64</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="5" unit="W/m²/µm">1287.61</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="6" unit="W/m²/µm">1162.08</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="7" unit="W/m²/µm">1041.63</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="8" unit="W/m²/µm">955.32</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="9" unit="W/m²/µm">812.92</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="10" unit="W/m²/µm">367.15</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="11" unit="W/m²/µm">245.59</SOLAR_IRRADIANCE>
          <SOLAR_IRRADIANCE bandId="12" unit="W/m²/µm">85.25</SOLAR_IRRADIANCE>
        </Solar_Irradiance_List>
      </Reflectance_Conversion>
            <Spectral_Information_List>
        <Spectral_Information bandId="0" physicalBand="B1">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">412</MIN>
            <MAX unit="nm">456</MAX>
            <CENTRAL unit="nm">442.7</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.001775742 0.004073061 0.003626143 0.003515199 0.005729163 0.003780292 0.002636732 0.001262113 0.001987583 0.001368913 0.001250444 0.000463454 0.000814293 0.001376431 0.001485086 0.001823735 0.001626817 0.004392062 0.029008099 0.11874593 0.32387506 0.57281921 0.71472749 0.76196778 0.78929702 0.80862387 0.81089382 0.82419876 0.85415811 0.87079088 0.88731097 0.92619924 0.98228149 1 0.9752382 0.93596338 0.88997148 0.85021048 0.82569453 0.78390239 0.61417422 0.33007109 0.12410831 0.04365694 0.014749595</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="1" physicalBand="B2">
          <RESOLUTION>10</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">456</MIN>
            <MAX unit="nm">533</MAX>
            <CENTRAL unit="nm">492.7</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.04255531 0.0722983 0.15374322 0.32799225 0.55336788 0.71011166 0.75285179 0.75232691 0.75668081 0.76326948 0.76239425 0.7852515 0.81546669 0.86179176 0.89282599 0.9195221 0.91900649 0.91315754 0.90035366 0.88989693 0.8823246 0.87606118 0.88429987 0.90695544 0.93232085 0.93947252 0.94383543 0.92204086 0.8860231 0.84743609 0.81251687 0.7823971 0.7731087 0.77209054 0.78742652 0.81217177 0.84605052 0.88767996 0.92793997 0.95069235 0.96573311 0.96938253 0.96570294 0.95832003 0.95405064 0.95178268 0.95699722 0.96556515 0.9770514 0.97709574 0.97436606 0.95903183 0.93506318 0.90190134 0.87165792 0.84402444 0.82280852 0.81536043 0.82057639 0.8395149 0.86992171 0.91526205 0.96067028 0.99163699 1 0.98356097 0.91130763 0.74018256 0.50395858 0.3050155 0.18004605 0.10738342 0.06593592 0.04207746 0.02662129 0.0143396 0.00265779 0.00081822</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="2" physicalBand="B3">
          <RESOLUTION>10</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">538</MIN>
            <MAX unit="nm">583</MAX>
            <CENTRAL unit="nm">559.8</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.01448181 0.03422251 0.07346335 0.15444843 0.31661425 0.55322279 0.74859406 0.84890306 0.89772216 0.9215368 0.92572845 0.91122688 0.88818924 0.86523756 0.84718187 0.8387572 0.84459081 0.86219653 0.88838714 0.92443236 0.96017974 0.98685516 1 0.9986008 0.98076472 0.94522089 0.8981778 0.85580323 0.81841734 0.78862048 0.76460653 0.74963745 0.75055111 0.76137888 0.78244479 0.79890086 0.81016957 0.81408886 0.77358596 0.62881065 0.40397555 0.21542098 0.10715281 0.04792877 0.01848693 0.00108588</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="3" physicalBand="B4">
          <RESOLUTION>10</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">646</MIN>
            <MAX unit="nm">684</MAX>
            <CENTRAL unit="nm">664.6</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00141521 0.02590238 0.11651178 0.39088616 0.74959342 0.94485805 0.98011173 0.99406309 1 0.99545475 0.99052772 0.97733476 0.94055988 0.87894956 0.81629384 0.77345952 0.75448766 0.75991531 0.7826343 0.8101689 0.83612975 0.86125424 0.88609106 0.91138767 0.93405146 0.95042063 0.9592573 0.96039555 0.95913395 0.95809013 0.95527459 0.94376465 0.89490799 0.74426308 0.476777 0.22960399 0.08009118 0.02617076 0.00415242</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="4" physicalBand="B5">
          <RESOLUTION>20</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">695</MIN>
            <MAX unit="nm">714</MAX>
            <CENTRAL unit="nm">704.1</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.02835786 0.12369337 0.39378774 0.76113071 0.97108502 0.99889523 1 0.99412258 0.98321789 0.96704093 0.94847389 0.92714833 0.90372458 0.88614713 0.86723745 0.79075319 0.58840332 0.26334833 0.05675422 0.00618833</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="5" physicalBand="B6">
          <RESOLUTION>20</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">731</MIN>
            <MAX unit="nm">749</MAX>
            <CENTRAL unit="nm">740.5</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00171088 0.05467153 0.25806676 0.64722098 0.89218999 0.90232877 0.91508768 0.94115846 0.96299993 0.97510481 0.9770217 0.98736251 1 0.98880277 0.97179916 0.90126739 0.60672391 0.20520227 0.0267569</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="6" physicalBand="B7">
          <RESOLUTION>20</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">769</MIN>
            <MAX unit="nm">797</MAX>
            <CENTRAL unit="nm">782.8</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00045899 0.0117201 0.05219715 0.16561733 0.36903355 0.63685453 0.86119638 0.97002897 0.99119602 0.99897921 1 0.97725155 0.92572385 0.86605804 0.81969611 0.79407674 0.79111029 0.80431552 0.81902721 0.82571292 0.82011829 0.79222195 0.72054559 0.58767794 0.41430355 0.23088817 0.09850282 0.02736551 0.00516235</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="7" physicalBand="B8">
          <RESOLUTION>10</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">760</MIN>
            <MAX unit="nm">907</MAX>
            <CENTRAL unit="nm">832.8</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00067259 0.00388856 0 0 0 0 0 0 0 0 0 0 0 0.00028956 0.00702964 0.01752391 0.03231111 0.05328661 0.08299885 0.12748502 0.19591065 0.30246323 0.43553954 0.57141637 0.69766701 0.80303852 0.89115744 0.95284584 0.98894161 1 0.98840653 0.96389216 0.94207967 0.93694643 0.94227343 0.95395718 0.96828896 0.97966549 0.9854444 0.98592681 0.98391181 0.97793903 0.97722771 0.97810609 0.98144486 0.98764558 0.98857708 0.9862422 0.98070921 0.97078624 0.95721089 0.93865821 0.91672388 0.89620759 0.872888 0.85160331 0.8246394 0.80078117 0.7823386 0.76360274 0.74962771 0.7387221 0.73079407 0.72271237 0.72507708 0.72563856 0.72304217 0.72229211 0.71616364 0.71159446 0.70826954 0.70157205 0.69924532 0.70093762 0.70692733 0.71824001 0.73124634 0.7484061 0.76818541 0.78394807 0.7968381 0.80260206 0.8045194 0.80240918 0.79699072 0.78920304 0.77691621 0.76518406 0.75119717 0.73700357 0.72262399 0.70412578 0.68410805 0.66474528 0.64736891 0.63005125 0.61564222 0.60249557 0.58988992 0.57993399 0.57136506 0.56094242 0.55235105 0.54568236 0.53958052 0.53510215 0.53093675 0.53016508 0.52984662 0.53036682 0.53211463 0.53271918 0.53246806 0.53331158 0.5319278 0.53051055 0.52951499 0.52996848 0.53253373 0.53705085 0.54235344 0.54912497 0.55523055 0.56011135 0.55767999 0.54821984 0.53144613 0.50763528 0.47811224 0.45092793 0.42798466 0.41051405 0.40039139 0.40087302 0.40829375 0.42086556 0.43007022 0.42456692 0.39136817 0.33009008 0.25720509 0.18189031 0.11650668 0.07031579 0.04275381 0.02593154 0.01574394 0.00394326</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="8" physicalBand="B8A">
          <RESOLUTION>20</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">837</MIN>
            <MAX unit="nm">881</MAX>
            <CENTRAL unit="nm">864.7</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00030097 0 0 0 0 0 0 0 0 0 0.00157217 0.00249886 0.01332037 0.02614866 0.05260479 0.10779709 0.22160755 0.39721628 0.60986885 0.81658883 0.9322445 0.97210033 0.97545482 0.97538048 0.97328205 0.97607828 0.98034955 0.98690928 0.99087465 0.99741818 0.99984673 0.99939141 0.99587928 0.99541228 1 0.99640762 0.92359433 0.74137684 0.48965971 0.25020643 0.11221246 0.04755984 0.02297815 0.01061438 0.00108149</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="9" physicalBand="B9">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">932</MIN>
            <MAX unit="nm">958</MAX>
            <CENTRAL unit="nm">945.1</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.01662953 0.06111857 0.17407094 0.38946454 0.6645915 0.87454114 0.93695988 0.96751014 0.9893391 0.9951269 1 0.97845762 0.98069118 0.9922335 0.98798379 0.99428313 0.98348041 0.97820013 0.95023367 0.95299604 0.92240308 0.85573828 0.70970227 0.46429542 0.21538427 0.06534121 0.01625596</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="10" physicalBand="B10">
          <RESOLUTION>60</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">1337</MIN>
            <MAX unit="nm">1412</MAX>
            <CENTRAL unit="nm">1373.5</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00024052 5.404e-05 3.052e-05 2.872e-05 7.632e-05 0.00010949 8.804e-05 0.00012356 0.00017424 0.0003317 0.00036891 0.0004467 0.00065919 0.0010913 0.00196903 0.00373668 0.00801754 0.01884719 0.04466732 0.10165546 0.20111776 0.34284841 0.50710992 0.6632068 0.78377143 0.86153862 0.91000261 0.94193255 0.96182259 0.97365119 0.98169786 0.98795826 0.99283342 0.99649788 0.99906011 1 0.99907734 0.99601604 0.9909083 0.98479854 0.97802142 0.97030114 0.96080954 0.94849765 0.93314108 0.91482336 0.8937997 0.86825426 0.83023193 0.76384193 0.65440009 0.50671604 0.35014737 0.21799972 0.12643091 0.06768988 0.0322709 0.013544 0.00544557 0.00237642 0.00111267 0.00053796 0.0003457 0.00017488 0.00021619 0.00019479 0.00010421 5.919e-05 5.109e-05 6.115e-05 5.527e-05 3.856e-05 3.147e-05 0.00012289 0.0001089 2.502e-05</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="11" physicalBand="B11">
          <RESOLUTION>20</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">1539</MIN>
            <MAX unit="nm">1682</MAX>
            <CENTRAL unit="nm">1613.7</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>6.79e-06 6.66e-06 8e-06 2.734e-05 3.685e-05 8.851e-05 0.00014522 0.00024812 0.00047627 0.00056335 0.00065326 0.00089835 0.00114664 0.00165604 0.00241611 0.00350246 0.00524274 0.0081538 0.01237062 0.0186097 0.02721853 0.03879155 0.05379167 0.07353187 0.09932758 0.1334178 0.18029249 0.24484994 0.32834511 0.42749961 0.53576798 0.64570396 0.74245998 0.81447017 0.85866596 0.87924777 0.88665266 0.888727 0.89105732 0.89725046 0.90632982 0.91627527 0.9263751 0.93515828 0.94226446 0.94739906 0.95131987 0.95416808 0.95635128 0.95813297 0.96062738 0.96344083 0.96577764 0.96818134 0.97104025 0.97343195 0.97597444 0.97865413 0.97994672 0.98064126 0.98094979 0.98143338 0.98123856 0.98068083 0.98033995 0.98101894 0.98268503 0.98507875 0.98777658 0.9903608 0.99202087 0.9933069 0.99256744 0.99044883 0.98717314 0.98353656 0.9800432 0.97617287 0.97253451 0.96977033 0.96762556 0.9662626 0.96572411 0.96592079 0.96729798 0.96975438 0.97337748 0.97862858 0.98345358 0.98765317 0.9919238 0.99554959 0.99767411 0.99866451 0.99941783 0.99930984 0.99885298 0.99913515 0.99973164 0.99973592 1 0.9998438 0.9967639 0.99175576 0.9859206 0.97887302 0.97029262 0.96135891 0.95379752 0.94709017 0.94228614 0.93919512 0.93616637 0.92889205 0.9129921 0.88158383 0.82602164 0.74412949 0.64281662 0.53483955 0.42772166 0.32439525 0.23488131 0.16445229 0.11056237 0.07271886 0.04634859 0.02949618 0.01941871 0.0133487 0.00934594 0.00654231 0.00487921 0.00341903 0.00249864 0.00196431 0.00142754 0.00105878 0.00049978 0.00022833 0.00015999 3.415e-05 4.517e-05 1.313e-05</VALUES>
          </Spectral_Response>
        </Spectral_Information>
        <Spectral_Information bandId="12" physicalBand="B12">
          <RESOLUTION>20</RESOLUTION>
          <Wavelength>
            <MIN unit="nm">2078</MIN>
            <MAX unit="nm">2320</MAX>
            <CENTRAL unit="nm">2202.4</CENTRAL>
          </Wavelength>
          <Spectral_Response>
            <STEP unit="nm">1</STEP>
            <VALUES>0.00063835 0.00102286 0.00288712 0.00399879 0.00658916 0.00765458 0.00799918 0.00853524 0.00929493 0.00999614 0.01096645 0.01208363 0.01335837 0.01501119 0.01711931 0.01977307 0.02332743 0.02765779 0.03320435 0.04020464 0.04886709 0.0596238 0.07315348 0.09050885 0.11143964 0.13686671 0.16776886 0.20341457 0.24281992 0.28484195 0.32711894 0.36834301 0.40794043 0.4447145 0.47647207 0.50303896 0.52524762 0.54328057 0.55717994 0.5685619 0.57895708 0.58860881 0.59881758 0.60990899 0.62128986 0.63421311 0.64847648 0.66363778 0.67997936 0.69609688 0.71189957 0.7269499 0.74124079 0.75734734 0.77201504 0.78552587 0.79818641 0.80962939 0.81965718 0.82855741 0.83668178 0.84440292 0.85106862 0.85321701 0.85471321 0.8561428 0.85778963 0.8594989 0.86142876 0.86322831 0.86511218 0.8672932 0.86967076 0.87427502 0.87856212 0.88241466 0.88590611 0.8894516 0.89320419 0.8966738 0.89987484 0.90257636 0.90481219 0.90550545 0.90564491 0.90548208 0.90513822 0.90476379 0.90406427 0.90332978 0.90274309 0.90235795 0.90196488 0.90340528 0.90429478 0.90529761 0.90642862 0.90807348 0.91010493 0.91293181 0.91556686 0.91842631 0.92128288 0.92431702 0.92719913 0.92972159 0.93190455 0.93412538 0.93588954 0.93707083 0.93762594 0.93828534 0.93763643 0.94042634 0.94250397 0.94324531 0.94301861 0.94210283 0.94061808 0.93841726 0.93665003 0.93524569 0.93301102 0.92686708 0.92104485 0.91547175 0.91100989 0.90828339 0.9072733 0.90817907 0.91115631 0.91617845 0.92284525 0.92059829 0.91947472 0.91947973 0.92126575 0.92451632 0.92772589 0.93196884 0.93676408 0.94147739 0.94679545 0.95119533 0.95443018 0.95704142 0.95972628 0.9625372 0.96485326 0.96603599 0.96664138 0.96630455 0.96545713 0.96484036 0.96365512 0.96169531 0.95944859 0.95732078 0.95513625 0.95355574 0.95273072 0.95217795 0.95172542 0.9521403 0.95263595 0.95405248 0.95707559 0.96063594 0.96421772 0.96830187 0.97268597 0.97741944 0.98289489 0.9871429 0.99073348 0.99398244 0.99678431 0.99875181 1 0.9999284 0.9991523 0.99712951 0.99388228 0.98968273 0.98373274 0.97621057 0.96780985 0.95833495 0.94842856 0.93818752 0.9277078 0.91702104 0.90597951 0.89384371 0.88165575 0.86861704 0.85460324 0.84058628 0.82598123 0.80948042 0.79182917 0.7724052 0.74907137 0.72031195 0.68815487 0.65125598 0.6100244 0.56600904 0.52095058 0.47464344 0.42924778 0.38584718 0.34208462 0.30067509 0.26317221 0.22770037 0.19571781 0.16808736 0.14467686 0.12482737 0.10823403 0.09439655 0.08235799 0.07149445 0.0626855 0.05498009 0.04818852 0.04285814 0.03859244 0.03494044 0.03199172 0.02958044 0.02741084 0.02556884 0.02395058 0.02166741 0.0191457 0.01632139 0.0109837 0.00736032 0.00649061 0.00469736 0.00205874</VALUES>
          </Spectral_Response>
        </Spectral_Information>
      </Spectral_Information_List>
            <PHYSICAL_GAINS bandId="0">4.10137842</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="1">3.75605469</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="2">4.18741753</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="3">4.52205376</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="4">5.20680393</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="5">4.8729478</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="6">4.5356737</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="7">6.16247757</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="8">5.13772343</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="9">8.53898524</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="10">55.10485389</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="11">35.30373192</PHYSICAL_GAINS>
            <PHYSICAL_GAINS bandId="12">106.24732599</PHYSICAL_GAINS>
            <Scene_Classification_List>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_NODATA</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>0</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_SATURATED_DEFECTIVE</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>1</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_DARK_FEATURE_SHADOW</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>2</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_CLOUD_SHADOW</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>3</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_VEGETATION</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>4</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_NOT_VEGETATED</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>5</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_WATER</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>6</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_UNCLASSIFIED</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>7</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_CLOUD_MEDIUM_PROBA</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>8</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_CLOUD_HIGH_PROBA</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>9</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_THIN_CIRRUS</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>10</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
                <Scene_Classification_ID>
                    <SCENE_CLASSIFICATION_TEXT>SC_SNOW_ICE</SCENE_CLASSIFICATION_TEXT>
                    <SCENE_CLASSIFICATION_INDEX>11</SCENE_CLASSIFICATION_INDEX>
                </Scene_Classification_ID>
            </Scene_Classification_List>
        </Product_Image_Characteristics>
    </n1:General_Info>
    <n1:Geometric_Info>
        <Product_Footprint>
            <Product_Footprint>
                <Global_Footprint>
                    <EXT_POS_LIST>40.64479480422486 115.81682739339685 40.65079881136531 117.1154430676197 39.66155122739065 117.11377991452629 39.655752572676114 115.83386830444628 40.64479480422486 115.81682739339685 </EXT_POS_LIST>
                </Global_Footprint>
            </Product_Footprint>
            <RASTER_CS_TYPE>POINT</RASTER_CS_TYPE>
            <PIXEL_ORIGIN>1</PIXEL_ORIGIN>
        </Product_Footprint>
        <Coordinate_Reference_System>
            <GEO_TABLES version="1">EPSG</GEO_TABLES>
            <HORIZONTAL_CS_TYPE>GEOGRAPHIC</HORIZONTAL_CS_TYPE>
        </Coordinate_Reference_System>
    </n1:Geometric_Info>
    <n1:Auxiliary_Data_Info>
        <GIPP_List>
      <GIPP_FILENAME type="GIP_INVLOC" version="0006">S2A_OPER_GIP_INVLOC_MPC__20171206T000000_V20150703T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_LREXTR" version="0001">S2A_OPER_GIP_LREXTR_MPC__20150605T094736_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_ATMIMA" version="0002">S2A_OPER_GIP_ATMIMA_MPC__20150605T094744_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_ATMSAD" version="0005">S2A_OPER_GIP_ATMSAD_MPC__20160729T000005_V20150703T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_BLINDP" version="0003">S2A_OPER_GIP_BLINDP_MPC__20150605T094736_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_CLOINV" version="0005">S2A_OPER_GIP_CLOINV_MPC__20210609T000005_V20210823T030000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_CLOPAR" version="0001">S2A_OPER_GIP_CLOPAR_MPC__20220120T000001_V20220125T022000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_CONVER" version="999">S2A_OPER_GIP_CONVER_MPC__20150710T131444_V20150627T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_DATATI" version="0007">S2A_OPER_GIP_DATATI_MPC__20151117T131048_V20150703T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_DECOMP" version="0000">S2A_OPER_GIP_DECOMP_MPC__20121031T075922_V19830101T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_EARMOD" version="0001">S2__OPER_GIP_EARMOD_MPC__20150605T094736_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_ECMWFP" version="0002">S2A_OPER_GIP_ECMWFP_MPC__20121031T075922_V19830101T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_G2PARA" version="0027">S2A_OPER_GIP_G2PARA_MPC__20231208T000027_V20231213T070000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_G2PARE" version="0002">S2A_OPER_GIP_G2PARE_MPC__20150605T094736_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_GEOPAR" version="0003">S2A_OPER_GIP_GEOPAR_MPC__20150605T094741_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_INTDET" version="0010">S2A_OPER_GIP_INTDET_MPC__20220120T000010_V20220125T022000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_JP2KPA" version="0006">S2A_OPER_GIP_JP2KPA_MPC__20220120T000006_V20220125T022000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_MASPAR" version="0009">S2A_OPER_GIP_MASPAR_MPC__20220120T000009_V20220125T022000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_OLQCPA" version="0000">S2A_OPER_GIP_OLQCPA_MPC__20220715T000042_V20220830T002500_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_PRDLOC" version="0012">S2A_OPER_GIP_PRDLOC_MPC__20180301T130000_V20180305T005000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_PROBAS" version="0510">S2A_OPER_GIP_PROBAS_MPC__20240305T000510_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2ABCA" version="0242">S2A_OPER_GIP_R2ABCA_MPC__20240315T121000_V20240319T003000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2BINN" version="0003">S2A_OPER_GIP_R2BINN_MPC__20150605T094803_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2CRCO" version="0003">S2A_OPER_GIP_R2CRCO_MPC__20151023T224715_V20150622T224715_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094742_V20150622T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094742_V20150622T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094742_V20150622T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094742_V20150622T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DECT" version="0003">S2A_OPER_GIP_R2DECT_MPC__20150605T094741_V20150622T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEFI" version="0003">S2A_OPER_GIP_R2DEFI_MPC__20150605T094741_V20150622T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094741_V20150622T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DENT" version="0003">S2A_OPER_GIP_R2DENT_MPC__20150605T094742_V20150622T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2DEPI" version="0023">S2A_OPER_GIP_R2DEPI_MPC__20230424T160000_V20230426T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EOB2" version="0014">S2A_OPER_GIP_R2EOB2_MPC__20190412T145327_V20190429T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2EQOG" version="0211">S2A_OPER_GIP_R2EQOG_MPC__20240315T121000_V20240319T003000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094741_V20150622T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2L2NC" version="0003">S2A_OPER_GIP_R2L2NC_MPC__20150605T094742_V20150622T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2NOMO" version="0004">S2A_OPER_GIP_R2NOMO_MPC__20150605T094803_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2PARA" version="0009">S2A_OPER_GIP_R2PARA_MPC__20221206T000009_V20221206T073000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2SWIR" version="0021">S2A_OPER_GIP_R2SWIR_MPC__20180406T000021_V20180604T100000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_R2WAFI" version="0003">S2A_OPER_GIP_R2WAFI_MPC__20150605T094742_V20150622T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_RESPAR" version="0001">S2A_OPER_GIP_RESPAR_MPC__20150605T094736_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_SPAMOD" version="0026">S2A_OPER_GIP_SPAMOD_MPC__20231122T110026_V20231123T010000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_TILPAR" version="0007">S2A_OPER_GIP_TILPAR_MPC__20151209T095117_V20150622T000000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B8A</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131049_V20150703T000000_21000101T000000_B03</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B08</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131048_V20150703T000000_21000101T000000_B01</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B11</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B10</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B06</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131049_V20150703T000000_21000101T000000_B04</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131049_V20150703T000000_21000101T000000_B02</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B05</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131051_V20150703T000000_21000101T000000_B12</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B09</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_VIEDIR" version="0005">S2A_OPER_GIP_VIEDIR_MPC__20151117T131050_V20150703T000000_21000101T000000_B07</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_L2ACSC" version="21003">S2__OPER_GIP_L2ACSC_MPC__20220121T000003_V20220125T022000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_L2ACAC" version="21003">S2__OPER_GIP_L2ACAC_MPC__20220121T000004_V20220125T022000_21000101T000000_B00</GIPP_FILENAME>
      <GIPP_FILENAME type="GIP_PROBA2" version="0510">S2__OPER_GIP_PROBA2_MPC__20231208T000510_V20231213T070000_21000101T000000_B00</GIPP_FILENAME>

</GIPP_List>
        <PRODUCTION_DEM_TYPE>CopernicusDEM30</PRODUCTION_DEM_TYPE>
        <IERS_BULLETIN_FILENAME>S2__OPER_AUX_UT1UTC_PDMC_20240404T000000_V20240405T000000_20250404T000000</IERS_BULLETIN_FILENAME>
        <GRI_List/>
        <ECMWF_DATA_REF>S2__OPER_AUX_ECMWFD_ADG__20240410T120000_V20240410T210000_20240412T150000</ECMWF_DATA_REF>
        <LUT_List>
            <LUT_FILENAME>None</LUT_FILENAME>
        </LUT_List>
        <SNOW_CLIMATOLOGY_MAP>GlobalSnowMap.tiff</SNOW_CLIMATOLOGY_MAP>
        <ESACCI_WaterBodies_Map>ESACCI-LC-L4-WB-Map-150m-P13Y-2000-v4.0.tif</ESACCI_WaterBodies_Map>
        <ESACCI_LandCover_Map>ESACCI-LC-L4-LCCS-Map-300m-P1Y-2015-v2.0.7.tif</ESACCI_LandCover_Map>
        <ESACCI_SnowCondition_Map_Dir>ESACCI-LC-L4-Snow-Cond-500m-MONTHLY-2000-2012-v2.4</ESACCI_SnowCondition_Map_Dir>
    </n1:Auxiliary_Data_Info>
    <n1:Quality_Indicators_Info>
        <Cloud_Coverage_Assessment>3.500058</Cloud_Coverage_Assessment>
        <Technical_Quality_Assessment>
            <DEGRADED_ANC_DATA_PERCENTAGE>0.0</DEGRADED_ANC_DATA_PERCENTAGE>
            <DEGRADED_MSI_DATA_PERCENTAGE>0</DEGRADED_MSI_DATA_PERCENTAGE>
        </Technical_Quality_Assessment>
        <Quality_Control_Checks>
            <Quality_Inspections>
                <quality_check checkType="FORMAT_CORRECTNESS">PASSED</quality_check>
                <quality_check checkType="GENERAL_QUALITY">PASSED</quality_check>
                <quality_check checkType="GEOMETRIC_QUALITY">PASSED</quality_check>
                <quality_check checkType="L2A_QUALITY">PASSED</quality_check>
                <quality_check checkType="RADIOMETRIC_QUALITY">PASSED</quality_check>
                <quality_check checkType="SENSOR_QUALITY">PASSED</quality_check>
            </Quality_Inspections>
            <Failed_Inspections/>
        </Quality_Control_Checks>
        <Image_Content_QI>
            <CLOUDY_PIXEL_OVER_LAND_PERCENTAGE>3.354197</CLOUDY_PIXEL_OVER_LAND_PERCENTAGE>
            <NODATA_PIXEL_PERCENTAGE>0.0</NODATA_PIXEL_PERCENTAGE>
            <SATURATED_DEFECTIVE_PIXEL_PERCENTAGE>0.0</SATURATED_DEFECTIVE_PIXEL_PERCENTAGE>
            <DARK_FEATURES_PERCENTAGE>8.675177</DARK_FEATURES_PERCENTAGE>
            <CLOUD_SHADOW_PERCENTAGE>0.268831</CLOUD_SHADOW_PERCENTAGE>
            <VEGETATION_PERCENTAGE>2.81222</VEGETATION_PERCENTAGE>
            <NOT_VEGETATED_PERCENTAGE>83.179593</NOT_VEGETATED_PERCENTAGE>
            <WATER_PERCENTAGE>0.992827</WATER_PERCENTAGE>
            <UNCLASSIFIED_PERCENTAGE>0.571295</UNCLASSIFIED_PERCENTAGE>
            <MEDIUM_PROBA_CLOUDS_PERCENTAGE>0.275278</MEDIUM_PROBA_CLOUDS_PERCENTAGE>
            <HIGH_PROBA_CLOUDS_PERCENTAGE>0.038401</HIGH_PROBA_CLOUDS_PERCENTAGE>
            <THIN_CIRRUS_PERCENTAGE>3.18638</THIN_CIRRUS_PERCENTAGE>
            <SNOW_ICE_PERCENTAGE>0.0</SNOW_ICE_PERCENTAGE>
            <RADIATIVE_TRANSFER_ACCURACY>0.0</RADIATIVE_TRANSFER_ACCURACY>
            <WATER_VAPOUR_RETRIEVAL_ACCURACY>0.0</WATER_VAPOUR_RETRIEVAL_ACCURACY>
            <AOT_RETRIEVAL_ACCURACY>0.0</AOT_RETRIEVAL_ACCURACY>
            <AOT_RETRIEVAL_METHOD>CAMS</AOT_RETRIEVAL_METHOD>
            <GRANULE_MEAN_AOT>0.392921</GRANULE_MEAN_AOT>
            <GRANULE_MEAN_WV>1.224094</GRANULE_MEAN_WV>
            <OZONE_SOURCE>AUX_ECMWFT</OZONE_SOURCE>
            <OZONE_VALUE>357.927923</OZONE_VALUE>
        </Image_Content_QI>
    </n1:Quality_Indicators_Info>
</n1:Level-2A_User_Product>
"""  # noqa

PROCESS_LEVELS = ["L1C", "oldL1C", "L2A"]
MTD_XMLS = [mtd_l1c_xml, mtd_l1c_old_xml, mtd_l2a_xml]
TILE_XMLS = [mtd_l1c_tile_xml, mtd_l1c_tile_xml, mtd_l1c_tile_xml]


def xml_builder(process_level, mask_saturated=True, band_name=None):
    """Build fake SAFE MTD/Tile XML."""
    from satpy.readers.msi_safe import SAFEMSIMDXML, SAFEMSITileMDXML
    filename_info = dict(observation_time=fname_dt, dtile_number=None, band_name=band_name, fmission_id="S2A",
                         process_level=process_level.replace("old", ""))
    xml_fh = SAFEMSIMDXML(StringIO(MTD_XMLS[PROCESS_LEVELS.index(process_level)]),
                          filename_info, mock.MagicMock(), mask_saturated=mask_saturated)
    xml_tile_fh = SAFEMSITileMDXML(BytesIO(TILE_XMLS[PROCESS_LEVELS.index(process_level)]),
                                   filename_info, mock.MagicMock())
    return xml_fh, xml_tile_fh


def jp2_builder(process_level, band_name, mask_saturated=True, test_l1b=False):
    """Build fake SAFE jp2 image file."""
    from satpy.readers.msi_safe import SAFEMSIL1C, SAFEMSITileMDXML
    filename_info = dict(observation_time=fname_dt, dtile_number=None, band_name=band_name, fmission_id="S2A",
                         process_level=process_level.replace("old", ""))
    if test_l1b:
        filename_info["process_level"] = "L1B"

    xml_fh = xml_builder(process_level, mask_saturated, band_name)[0]
    tile_xml_fh = mock.create_autospec(SAFEMSITileMDXML)(BytesIO(TILE_XMLS[PROCESS_LEVELS.index(process_level)]),
                                                         filename_info, mock.MagicMock())
    tile_xml_fh.start_time.return_value = tilemd_dt
    tile_xml_fh.get_dataset.return_value = xr.DataArray(da.from_array([[22.5, 23.8], [22.5, 24.8]]),
                                                        dims=["x", "y"])
    jp2_fh = SAFEMSIL1C("somefile", filename_info, mock.MagicMock(), xml_fh, tile_xml_fh)
    return jp2_fh


def make_alt_dataid(**items):
    """Make a DataID with modified keys."""
    from satpy.dataset.dataid import DataID, ModifierTuple, WavelengthRange
    modified_id_keys_config = {
        "name": {
            "required": True,
        },
        "wavelength": {
            "type": WavelengthRange,
        },
        "resolution": {
            "transitive": False,
        },
        "calibration": {
            "enum": [
                "reflectance",
                "brightness_temperature",
                "radiance",
                "radiance_wavenumber",
                "counts",
                "aerosol_thickness",
                "water_vapor"
            ],
            "transitive": True,
        },
        "modifiers": {
            "default": ModifierTuple(),
            "type": ModifierTuple,
        },
    }

    return DataID(modified_id_keys_config, **items)


class TestTileXML:
    """Test the SAFE TILE XML file handler.

    Since L1C/L2A share almost the same Tile XML structure, we only use L1C Tile here.

    """

    @pytest.mark.parametrize(("process_level", "angle_name", "angle_tag", "expected"),
                             [
                                 ("L1C", "satellite_zenith_angle", ("Viewing_Incidence_Angles_Grids", "Zenith"),
                                  [[11.7128, 11.18397802, 10.27667671, 9.35384969, 8.42850504,
                                    7.55445611, 6.65475545, 5.66517232, 4.75893757, 4.04976844],
                                   [11.88606009, 10.9799713, 10.07083278, 9.14571825, 8.22607131,
                                    7.35181457, 6.44647222, 5.46144173, 4.56625547, 3.86638233],
                                   [11.6823579, 10.7763071, 9.86302106, 8.93879112, 8.04005637,
                                    7.15028077, 6.21461062, 5.25780953, 4.39876601, 3.68620793],
                                   [11.06724679, 10.35723901, 9.63958896, 8.73072512, 7.83680864,
                                    6.94792574, 5.9889201, 5.05445872, 4.26089708, 3.50984272],
                                   [6.28411038, 6.28411038, 6.28411038, 6.28411038, 6.28411038,
                                    5.99769643, 5.62586167, 4.85165966, 4.13238314, 3.33781401],
                                   [3.7708, 3.7708, 3.7708, 3.7708, 3.7708,
                                    3.7708, 3.7708, 3.7708, 3.7708, 3.24140837],
                                   [3.7708, 3.7708, 3.7708, 3.7708, 3.7708,
                                    3.7708, 3.7708, 3.7708, 3.7708, 3.24140837],
                                   [3.7708, 3.7708, 3.7708, 3.7708, 3.7708,
                                    3.7708, 3.7708, 3.7708, 3.7708, 3.24140837],
                                   [3.7708, 3.7708, 3.7708, 3.7708, 3.7708,
                                    3.7708, 3.7708, 3.7708, 3.7708, 3.24140837],
                                   [3.7708, 3.7708, 3.7708, 3.7708, 3.7708,
                                    3.7708, 3.7708, 3.7708, 3.7708, 3.24140837]]),
                                 ("L2A", "solar_zenith_angle", ("Sun_Angles_Grid", "Zenith"),
                                  [[39.8824, 39.83721367, 39.79230847, 39.74758442, 39.7030415,
                                    39.65867687, 39.61455566, 39.57061558, 39.52685664, 39.48331372],
                                   [39.78150175, 39.73629896, 39.69128852, 39.64643679, 39.6018404,
                                    39.5574369, 39.51323286, 39.46920212, 39.4253673, 39.38179377],
                                   [39.6806035, 39.63532838, 39.5902497, 39.54538507, 39.5007087,
                                    39.45621756, 39.41195347, 39.36779169, 39.3239121, 39.28027381],
                                   [39.57980525, 39.53445664, 39.48931088, 39.44434154, 39.39957879,
                                    39.35503587, 39.31067408, 39.26649344, 39.22249393, 39.17876143],
                                   [39.479007, 39.43355483, 39.38829092, 39.34328573, 39.29846167,
                                    39.25381983, 39.2093947, 39.16513007, 39.12109926, 39.07726878],
                                   [39.37820875, 39.33268069, 39.28735495, 39.24224914, 39.19736058,
                                    39.15267709, 39.1081719, 39.06385068, 39.01973446, 38.97584982],
                                   [39.2774105, 39.23184303, 39.18646737, 39.14130809, 39.09632176,
                                    39.05153988, 39.00696049, 38.9625713, 38.91842056, 38.87444401],
                                   [39.17671225, 39.13104478, 39.08559031, 39.04034757, 38.99528294,
                                    38.95039991, 38.9057971, 38.86130793, 38.81705183, 38.77303821],
                                   [39.076014, 39.03026112, 38.98477906, 38.93940875, 38.89425338,
                                    38.84936063, 38.80464763, 38.76011645, 38.7157479, 38.67164839],
                                   [38.97531575, 38.92950771, 38.88389967, 38.83852091, 38.7933053,
                                    38.74831897, 38.7034912, 38.65891427, 38.61446851, 38.57030388]]),
                                 ("L1C", "moon_zenith_angle", ("Sun_Angles_Grid", "Zenith"), None)
                             ])
    def test_angles(self, process_level, angle_name, angle_tag, expected):
        """Test reading angles array."""
        info = dict(xml_tag=angle_tag[0], xml_item=angle_tag[1]) if "satellite" in angle_name else \
            dict(xml_tag=angle_tag[0] + "/" + angle_tag[1])
        xml_tile_fh = xml_builder(process_level)[1]

        with warnings.catch_warnings():
            # L1C case includes NaNs in angles that result in NaN during the nanmean calculation
            warnings.filterwarnings("ignore", message="Mean of empty slice", category=RuntimeWarning)
            res = xml_tile_fh.get_dataset(make_alt_dataid(name=angle_name, resolution=60), info)
        if res is not None:
            res = res[::200, ::200]

        if res is not None:
            np.testing.assert_allclose(res, expected)
        else:
            assert res is expected

    def test_start_time(self):
        """Ensure start time is read correctly from XML."""
        xml_tile_fh = xml_builder("L1C")[1]
        assert xml_tile_fh.start_time() == tilemd_dt

    def test_navigation(self):
        """Test the navigation."""
        from pyproj import CRS
        crs = CRS("EPSG:32616")

        dsid = make_alt_dataid(name="B01", resolution=60)
        xml_tile_fh = xml_builder("L1C")[1]
        result = xml_tile_fh.get_area_def(dsid)
        area_extent = (499980.0, 3590220.0, 609780.0, 3700020.0)
        assert result.crs == crs
        np.testing.assert_allclose(result.area_extent, area_extent)


class TestMTDXML:
    """Test the SAFE MTD XML file handler."""

    def setup_method(self):
        """Set up the test case."""
        self.fake_data = xr.DataArray(da.from_array([[[0, 1, 2, 3], [4, 1000, 65534, 65535]]]), dims=["band", "x", "y"])

    @pytest.mark.parametrize(("process_level", "mask_saturated", "band_name", "expected"),
                             [
                                 ("L1C", True, "B01", ([[[np.nan, -9.99, -9.98, -9.97],
                                                         [-9.96, 0, 645.34, np.inf]]],
                                                       [[[0.0, 5.26972293, 10.5394459, 15.8091688,],
                                                         [21.0788917, 5269.72293, 345346.023, 345351.292]]],
                                                       [[[np.nan, 1, 2, 3],
                                                         [4, 1000, 65534, np.inf]]])),
                                 ("L1C", False, "B10", ([[[np.nan, -19.99, -19.98, -19.97],
                                                          [-19.96, -10, 635.34, 635.35]]],
                                                        [[[0., 1.027375, 2.054751, 3.082126],
                                                          [4.109501, 1027.375264, 67328.010528, 67329.037903]]],
                                                        [[[np.nan, 1, 2, 3],
                                                          [4, 1000, 65534, 65535]]])),
                                 ("oldL1C", True, "B01", ([[[np.nan, 0.01, 0.02, 0.03],
                                                            [0.04, 10, 655.34, np.inf]]],
                                                          [[[0., 5.269723, 10.539446, 15.809169,],
                                                            [21.078892, 5269.722932, 345346.02264, 345351.292363]]],
                                                          [[[np.nan, 1, 2, 3],
                                                            [4, 1000, 65534, np.inf]]])),
                                 ("L2A", False, "B03", ([[[np.nan, -9.99, -9.98, -9.97],
                                                          [-9.96, 0, 645.34, 645.35]]],
                                                        [[[0., 5.224829, 10.449658, 15.674487,],
                                                          [20.899316, 5224.8289, 342403.937147, 342409.161976]]],
                                                        [[[np.nan, 1, 2, 3],
                                                          [4, 1000, 65534, 65535]]])),
                             ])
    def test_xml_calibration(self, process_level, mask_saturated, band_name, expected):
        """Test the calibration to reflectance/radiance/counts."""
        xml_fh = xml_builder(process_level, mask_saturated)[0]

        res1 = xml_fh.calibrate_to_reflectances(self.fake_data, band_name)
        res2 = xml_fh.calibrate_to_radiances(self.fake_data, 25.6, band_name)
        res3 = xml_fh._sanitize_data(self.fake_data)

        results = (res1, res2, res3)

        np.testing.assert_allclose(results, expected, rtol=1e-6, atol=1e-5)

    @pytest.mark.parametrize(("process_level", "mask_saturated", "band_name", "expected"),
                             [
                                 ("L1C", True, "B01", None),
                                 ("L2A", False, "AOT", [[[np.nan, 0.001, 0.002, 0.003],
                                                         [0.004, 1., 65.534, 65.535]]]),
                                 ("L2A", True, "WVP", [[[np.nan, 0.001, 0.002, 0.003],
                                                        [0.004, 1., 65.534, np.inf]]]),
                                 ("L2A", False, "CLOUD", None),
                                 ("L2A", False, "B10", None),
                             ])
    def test_xml_calibration_to_atmospheric(self, process_level, mask_saturated, band_name, expected):
        """Test the calibration to L2A atmospheric products."""
        xml_fh = xml_builder(process_level, mask_saturated)[0]

        result = xml_fh.calibrate_to_atmospheric(self.fake_data, band_name)

        if result is not None:
            np.testing.assert_allclose(result, expected)
        else:
            assert result is expected


class TestSAFEMSIL1C:
    """Test case for image reading (jp2k)."""

    def setup_method(self):
        """Set up the test."""
        self.fake_data = xr.Dataset(
            {
                "band_data": xr.DataArray(
                    da.from_array([[[0, 1], [65534, 65535]]]),
                    dims=["band", "x", "y"],
                ),
            },
        )
        self.fake_data_l1b = xr.Dataset(
            {
                "band_data": xr.DataArray(
                    da.from_array([[[1000, 1205.5], [3000.4, 2542.]]]),
                    dims=["band", "x", "y"],
                ),
            },
        )

    @pytest.mark.parametrize(("process_level", "mask_saturated", "dataset_name", "calibration", "expected"),
                             [
                                 ("L2A", False, "B01", "reflectance", [[np.nan, -9.99], [645.34, 645.35]]),
                                 ("L1C", True, "B02", "radiance", [[np.nan, -55.845848], [3642.73338, np.inf]]),
                                 ("L2A", True, "B03", "counts", [[np.nan, 1], [65534, np.inf]]),
                                 ("L2A", False, "AOT", "aerosol_thickness", [[np.nan, 0.001], [65.534, 65.535]]),
                                 ("L2A", True, "WVP", "water_vapor", [[np.nan, 0.001], [65.534, np.inf]]),
                                 ("L2A", True, "SNOW", "water_vapor", None),
                             ])
    def test_calibration_and_masking(self, process_level, mask_saturated, dataset_name, calibration, expected):
        """Test that saturated is masked with inf when requested and that calibration is performed."""
        jp2_fh = jp2_builder(process_level, dataset_name, mask_saturated)

        with mock.patch("xarray.open_dataset", return_value=self.fake_data):
            res = jp2_fh.get_dataset(make_alt_dataid(name=dataset_name, calibration=calibration, resolution="20"),
                                     info=dict())
            if res is not None:
                np.testing.assert_allclose(res, expected)
            else:
                assert res is expected

    @pytest.mark.parametrize(("process_level", "band_name", "dataset_name"),
                             [
                                 ("L1C", "B01", "B03"),
                                 ("L2A", "B02", "B03"),
                             ])
    def test_filename_dsname_mismatch(self, process_level, band_name, dataset_name):
        """Test when dataset name and file band name mismatch, the data and its area definition should both be None."""
        jp2_fh = jp2_builder(process_level, band_name)

        with mock.patch("xarray.open_dataset", return_value=self.fake_data):
            res1 = jp2_fh.get_dataset(make_alt_dataid(name=dataset_name), info=dict())
            res2 = jp2_fh.get_area_def(make_alt_dataid(name=dataset_name))

            assert res1 is None
            assert res2 is None

    def test_start_end_time(self):
        """Test that the correct start time is returned."""
        jp2_fh = jp2_builder("L1C", "B01")
        assert tilemd_dt == jp2_fh.start_time
        assert tilemd_dt == jp2_fh.end_time

    def test_l1b_error(self):
        """We can't process L1B data yet, so check an error is raised."""
        with pytest.raises(ValueError, match="Unsupported process level: L1B"):
            jp2_builder("L1C", "B01", test_l1b=True)


    @pytest.mark.parametrize(("st_str", "en_str", "err_str"),
                             [
                                 ("<U>",
                                  "</U>",
                                  "Solar correction factor, U, in metadata is missing."),
                                 ("<Solar_Irradiance_List>",
                                  "</Solar_Irradiance_List>",
                                  "No solar irradiance values were found in the metadata."),
                             ])
    def test_missing_esd(self, st_str, en_str, err_str):
        """Test that missing Solar Correction factor in the metadata is handled correctly."""
        from satpy.readers.msi_safe import SAFEMSIMDXML

        tmp_xml = str(mtd_l1c_xml)
        p1 = tmp_xml.find(st_str)
        p2 = tmp_xml.find(en_str)
        tmp_xml = tmp_xml[:p1+len(st_str)] + tmp_xml[p2:]

        filename_info = dict(observation_time=fname_dt, dtile_number=None,
                             band_name="B01", fmission_id="S2A", process_level="L1C")

        xml_fh = SAFEMSIMDXML(StringIO(tmp_xml), filename_info, mock.MagicMock())

        if st_str == "<U>":
            with pytest.raises(ValueError, match=err_str):
                xml_fh.solar_correction_factor
        else:
            with pytest.raises(ValueError, match=err_str):
                xml_fh.solar_irradiances


    def test_l1b_calib(self):
        """Test that Level-1B calibration can be performed."""
        from satpy.readers.msi_safe import SAFEMSIMDXML

        filename_info = dict(observation_time=fname_dt, dtile_number=None,
                             band_name="B01", fmission_id="S2A", process_level="L1C")

        xml_fh = SAFEMSIMDXML(StringIO(mtd_l1c_xml), filename_info, mock.MagicMock())

        res = xml_fh.calibrate_to_radiances_l1b(self.fake_data_l1b, "B01")
        np.testing.assert_allclose(res.band_data.data.ravel(),
                                   np.array((0.0, 51.752319, 503.77294, 388.33127)),
                                   rtol=1e-4)
