// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;

using Internal.IL;
using Internal.TypeSystem;
using Internal.Text;
using ILCompiler.DependencyAnalysis;

using Debug = System.Diagnostics.Debug;

namespace ILCompiler
{
    /// <summary>
    /// Captures information required to generate a ReadyToRun helper to create a delegate type instance
    /// pointing to a specific target method.
    /// </summary>
    public sealed class DelegateCreationInfo
    {
        private enum TargetKind
        {
            CanonicalEntrypoint,
            ExactCallableAddress,
            InterfaceDispatch,
            VTableLookup,
            MethodHandle,
        }

        private TargetKind _targetKind;

        /// <summary>
        /// Gets the node corresponding to the method that initializes the delegate.
        /// </summary>
        public IMethodNode Constructor
        {
            get;
        }

        public MethodDesc TargetMethod
        {
            get;
        }

        private bool TargetMethodIsUnboxingThunk
        {
            get
            {
                return TargetMethod.OwningType.IsValueType && !TargetMethod.Signature.IsStatic;
            }
        }

        public bool TargetNeedsVTableLookup => _targetKind == TargetKind.VTableLookup;

        public bool NeedsVirtualMethodUseTracking
        {
            get
            {
                return _targetKind == TargetKind.VTableLookup || _targetKind == TargetKind.InterfaceDispatch;
            }
        }

        public bool NeedsRuntimeLookup
        {
            get
            {
                switch (_targetKind)
                {
                    case TargetKind.VTableLookup:
                        return false;

                    case TargetKind.CanonicalEntrypoint:
                    case TargetKind.ExactCallableAddress:
                    case TargetKind.InterfaceDispatch:
                    case TargetKind.MethodHandle:
                        return TargetMethod.IsRuntimeDeterminedExactMethod;

                    default:
                        Debug.Assert(false);
                        return false;
                }
            }
        }

        // None of the data structures that support shared generics have been ported to the JIT
        // codebase which makes this a huge PITA. Not including the method for JIT since nobody
        // uses it in that mode anyway.
#if !SUPPORT_JIT
        public GenericLookupResult GetLookupKind(NodeFactory factory)
        {
            Debug.Assert(NeedsRuntimeLookup);
            switch (_targetKind)
            {
                case TargetKind.ExactCallableAddress:
                    return factory.GenericLookup.MethodEntry(TargetMethod, TargetMethodIsUnboxingThunk);

                case TargetKind.InterfaceDispatch:
                    return factory.GenericLookup.VirtualDispatchCell(TargetMethod);

                case TargetKind.MethodHandle:
                    return factory.GenericLookup.MethodHandle(TargetMethod);

                default:
                    Debug.Assert(false);
                    return null;
            }
        }
#endif

        /// <summary>
        /// Gets the node representing the target method of the delegate if no runtime lookup is needed.
        /// </summary>
        public ISymbolNode GetTargetNode(NodeFactory factory)
        {
            Debug.Assert(!NeedsRuntimeLookup);
            switch (_targetKind)
            {
                case TargetKind.CanonicalEntrypoint:
                    return factory.CanonicalEntrypoint(TargetMethod, TargetMethodIsUnboxingThunk);

                case TargetKind.ExactCallableAddress:
                    return factory.ExactCallableAddress(TargetMethod, TargetMethodIsUnboxingThunk);

                case TargetKind.InterfaceDispatch:
                    return factory.InterfaceDispatchCell(TargetMethod);

                case TargetKind.MethodHandle:
                    return factory.RuntimeMethodHandle(TargetMethod);

                case TargetKind.VTableLookup:
                    Debug.Fail("Need to do runtime lookup");
                    return null;

                default:
                    Debug.Assert(false);
                    return null;
            }
        }

        /// <summary>
        /// Gets an optional node passed as an additional argument to the constructor.
        /// </summary>
        public IMethodNode Thunk
        {
            get;
        }

        private DelegateCreationInfo(IMethodNode constructor, MethodDesc targetMethod, TargetKind targetKind, IMethodNode thunk = null)
        {
            Constructor = constructor;
            TargetMethod = targetMethod;
            _targetKind = targetKind;
            Thunk = thunk;
        }

        /// <summary>
        /// Constructs a new instance of <see cref="DelegateCreationInfo"/> set up to construct a delegate of type
        /// '<paramref name="delegateType"/>' pointing to '<paramref name="targetMethod"/>'.
        /// </summary>
        public static DelegateCreationInfo Create(TypeDesc delegateType, MethodDesc targetMethod, NodeFactory factory, bool followVirtualDispatch)
        {
            TypeSystemContext context = delegateType.Context;
            DefType systemDelegate = context.GetWellKnownType(WellKnownType.MulticastDelegate).BaseType;

            int paramCountTargetMethod = targetMethod.Signature.Length;
            if (!targetMethod.Signature.IsStatic)
            {
                paramCountTargetMethod++;
            }

            DelegateInfo delegateInfo = context.GetDelegateInfo(delegateType.GetTypeDefinition());
            int paramCountDelegateClosed = delegateInfo.Signature.Length + 1;
            bool closed = false;
            if (paramCountDelegateClosed == paramCountTargetMethod)
            {
                closed = true;
            }
            else
            {
                Debug.Assert(paramCountDelegateClosed == paramCountTargetMethod + 1);
            }

            if (targetMethod.Signature.IsStatic)
            {
                MethodDesc invokeThunk;
                MethodDesc initMethod;

                if (!closed)
                {
                    // Open delegate to a static method
                    if (targetMethod.IsNativeCallable)
                    {
                        // If target method is native callable, create a reverse PInvoke delegate
                        initMethod = systemDelegate.GetKnownMethod("InitializeReversePInvokeThunk", null);
                        invokeThunk = delegateInfo.Thunks[DelegateThunkKind.ReversePinvokeThunk];

                        // You might hit this when the delegate is generic: you need to make the delegate non-generic.
                        // If the code works on Project N, it's because the delegate is used in connection with
                        // AddrOf intrinsic (please validate that). We don't have the necessary AddrOf expansion in
                        // the codegen to make this work without actually constructing the delegate. You can't construct
                        // the delegate if it's generic, even on Project N.
                        // TODO: Make this throw something like "TypeSystemException.InvalidProgramException"?
                        Debug.Assert(invokeThunk != null, "Delegate with a non-native signature for a NativeCallable method");
                    }
                    else
                    {
                        initMethod = systemDelegate.GetKnownMethod("InitializeOpenStaticThunk", null);
                        invokeThunk = delegateInfo.Thunks[DelegateThunkKind.OpenStaticThunk];
                    }
                }
                else
                {
                    // Closed delegate to a static method (i.e. delegate to an extension method that locks the first parameter)
                    invokeThunk = delegateInfo.Thunks[DelegateThunkKind.ClosedStaticThunk];
                    initMethod = systemDelegate.GetKnownMethod("InitializeClosedStaticThunk", null);
                }

                var instantiatedDelegateType = delegateType as InstantiatedType;
                if (instantiatedDelegateType != null)
                    invokeThunk = context.GetMethodForInstantiatedType(invokeThunk, instantiatedDelegateType);

                return new DelegateCreationInfo(
                    factory.MethodEntrypoint(initMethod),
                    targetMethod,
                    TargetKind.ExactCallableAddress,
                    factory.MethodEntrypoint(invokeThunk));
            }
            else
            {
                if (!closed)
                    throw new NotImplementedException("Open instance delegates");

                string initializeMethodName = "InitializeClosedInstance";
                MethodDesc targetCanonMethod = targetMethod.GetCanonMethodTarget(CanonicalFormKind.Specific);
                TargetKind kind;
                if (targetMethod.HasInstantiation)
                {
                    if (followVirtualDispatch && targetMethod.IsVirtual)
                    {
                        initializeMethodName = "InitializeClosedInstanceWithGVMResolution";
                        kind = TargetKind.MethodHandle;
                    }
                    else
                    {
                        if (targetMethod != targetCanonMethod)
                        {
                            // Closed delegates to generic instance methods need to be constructed through a slow helper that
                            // checks for the fat function pointer case (function pointer + instantiation argument in a single
                            // pointer) and injects an invocation thunk to unwrap the fat function pointer as part of
                            // the invocation if necessary.
                            initializeMethodName = "InitializeClosedInstanceSlow";
                        }

                        kind = TargetKind.ExactCallableAddress;
                    }
                }
                else
                {
                    if (followVirtualDispatch && targetMethod.IsVirtual)
                    {
                        if (targetMethod.OwningType.IsInterface)
                        {
                            kind = TargetKind.InterfaceDispatch;
                            initializeMethodName = "InitializeClosedInstanceToInterface";
                        }
                        else
                        {
                            kind = TargetKind.VTableLookup;
                            targetMethod = targetMethod.GetCanonMethodTarget(CanonicalFormKind.Specific);
                        }
                    }
                    else
                    {
                        kind = TargetKind.CanonicalEntrypoint;
                        targetMethod = targetMethod.GetCanonMethodTarget(CanonicalFormKind.Specific);
                    }
                }

                return new DelegateCreationInfo(
                    factory.MethodEntrypoint(systemDelegate.GetKnownMethod(initializeMethodName, null)),
                    targetMethod,
                    kind);
            }
        }

        public void AppendMangledName(NameMangler nameMangler, Utf8StringBuilder sb)
        {
            sb.Append("__DelegateCtor_");
            if (TargetNeedsVTableLookup)
                sb.Append("FromVtbl_");
            Constructor.AppendMangledName(nameMangler, sb);
            sb.Append("__");
            sb.Append(nameMangler.GetMangledMethodName(TargetMethod));
            if (Thunk != null)
            {
                sb.Append("__");
                Thunk.AppendMangledName(nameMangler, sb);
            }
        }

        public override bool Equals(object obj)
        {
            var other = obj as DelegateCreationInfo;
            return other != null
                && Constructor == other.Constructor
                && TargetMethod == other.TargetMethod
                && _targetKind == other._targetKind
                && Thunk == other.Thunk;
        }

        public override int GetHashCode()
        {
            return Constructor.GetHashCode() ^ TargetMethod.GetHashCode();
        }

#if !SUPPORT_JIT
        internal int CompareTo(DelegateCreationInfo other, TypeSystemComparer comparer)
        {
            var compare = _targetKind - other._targetKind;
            if (compare != 0)
                return compare;

            compare = comparer.Compare(TargetMethod, other.TargetMethod);
            if (compare != 0)
                return compare;

            compare = comparer.Compare(Constructor.Method, other.Constructor.Method);
            if (compare != 0)
                return compare;

            if (Thunk == other.Thunk)
                return 0;

            if (Thunk == null)
                return -1;

            if (other.Thunk == null)
                return 1;

            return comparer.Compare(Thunk.Method, other.Thunk.Method);
        }
#endif
    }
}
