use sql;
use test;

// Initial state:
DATABASE Migrate1 {
	TABLE test(
		id INTEGER PRIMARY KEY,
		b INTEGER(6), // Ask for an 8-byte integer
		c TEXT(32) DEFAULT "c", // Note: MySQL/MariaDB has a limit on the lenght of things that are indexed.
		d TEXT DEFAULT "d"
		);

	INDEX ON test(b);
}

// First migration:
DATABASE Migrate2 {
	TABLE test(
		id INTEGER,
		b INTEGER(6),
		c TEXT(32) ALLOW NULL,
		d TEXT UNIQUE DEFAULT "c",
		added INTEGER ALLOW NULL, // TODO: Check what happens if we don't allow null nor specify a default!
		PRIMARY KEY(id, b)
		);

	INDEX ON test(b, c);
}

// Second migration:
DATABASE Migrate3 {
	TABLE test(
		id INTEGER PRIMARY KEY,
		d TEXT,
		b INTEGER(6),
		added INTEGER ALLOW NULL
		);
}

// Versions that don't bother with modifying attributes.
DATABASE Migrate2Simple {
	TABLE test(
		id INTEGER PRIMARY KEY,
		b INTEGER(6),
		c TEXT DEFAULT "c",
		d TEXT DEFAULT "d",
		added INTEGER ALLOW NULL
		);

	INDEX ON test(b, c);
}
DATABASE Migrate3Simple {
	TABLE test(
		id INTEGER PRIMARY KEY,
		d TEXT DEFAULT "d",
		b INTEGER(6),
		added INTEGER ALLOW NULL
		);
}

Bool verify(Str expected, Str actual) {
	if (expected != actual.trimWhitespace) {
		print("Expected:\n${expected}\nGot:\n${actual}");
		return false;
	} else {
		return true;
	}
}

Bool doMigrationsI(DBConnection db) {
	MigrationPolicy policy = MigrationPolicy:allowReorder + MigrationPolicy:allowRemoval;

	// Clear out the db beforehand to make it easier to run on persistent databases.
	var prepared = db.prepare("DROP TABLE IF EXISTS test");
	prepared.execute();
	prepared.finalize();

	{
		Migrate1 t(db, policy);
		WITH t: INSERT INTO test(b) VALUES (1);
		WITH t: INSERT INTO test(b, c, d) VALUES (2, "two", "three");
	}

	if (!db.features.has(DBFeatures:limitedUpdate)) {

		{
			Migrate2 t(db, policy);
			// TODO: We should try to insert something as well.
			StrBuf out;
			for (row in WITH t: SELECT * FROM test) {
				out << row << "\n";
			}
			var expected = str{
				{ id: 1, b: 1, c: c, d: d, added: null }
				{ id: 2, b: 2, c: two, d: three, added: null }
			};
			if (!verify(expected, out.toS))
				return false;
		}

		{
			Migrate3 t(db, policy);
			StrBuf out;
			for (row in WITH t: SELECT * FROM test) {
				out << row << "\n";
			}
			var expected = str{
				{ id: 1, d: d, b: 1, added: null }
				{ id: 2, d: three, b: 2, added: null }
			};
			if (!verify(expected, out.toS))
				return false;
		}

	} else {

		{
			Migrate2Simple t(db, policy);
			StrBuf out;
			for (row in WITH t: SELECT * FROM test) {
				out << row << "\n";
			}
			var expected = str{
				{ id: 1, b: 1, c: c, d: d, added: null }
				{ id: 2, b: 2, c: two, d: three, added: null }
			};
			if (!verify(expected, out.toS))
				return false;
		}

		{
			Migrate3Simple t(db, policy);
			StrBuf out;
			for (row in WITH t: SELECT * FROM test) {
				out << row << "\n";
			}
			var expected = str{
				{ id: 1, d: d, b: 1, added: null }
				{ id: 2, d: three, b: 2, added: null }
			};
			if (!verify(expected, out.toS))
				return false;
		}

	}

	true;
}

Bool doMigrations(DBConnection db) {
	try {
		doMigrationsI(db);
	} catch (Exception e) {
		print("Error: ${e}");
		db.close();
		throw e;
	}
}

// Test on MariaDB - requires a local database, so it is not a "real" test.
void migrateMariaDB() {
	if (doMigrations(MariaDB(Host:local(), null, null, "test")))
		print("OK!");
	else
		print("Failed");
}

// Test on PostgreSQL - requires a local database, so it is not a "real" test.
void migratePostgreSQL() {
	if (doMigrations(PostgreSQL(Host:local(), null, null, "test")))
		print("OK!");
	else
		print("Failed");
}

test AutoMigrateComplex {
	// This is the only DB we can use without additional setup.
	check doMigrations(SQLite());
}
