/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-get-pot-ID.c
 * @brief implementation of GET /private/pots/$POT_ID
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-get-pot-ID.h"
#include <taler/taler_json_lib.h>


MHD_RESULT
TMH_private_get_pot (const struct TMH_RequestHandler *rh,
                     struct MHD_Connection *connection,
                     struct TMH_HandlerContext *hc)
{
  const char *pot_id_str = hc->infix;
  unsigned long long pot_id;
  char *pot_name;
  char *description;
  size_t pot_total_len;
  struct TALER_Amount *pot_totals;
  enum GNUNET_DB_QueryStatus qs;
  char dummy;

  (void) rh;
  if (1 != sscanf (pot_id_str,
                   "%llu%c",
                   &pot_id,
                   &dummy))
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "pot_id");
  }
  qs = TMH_db->select_money_pot (TMH_db->cls,
                                 hc->instance->settings.id,
                                 pot_id,
                                 &pot_name,
                                 &description,
                                 &pot_total_len,
                                 &pot_totals);

  if (qs < 0)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_FETCH_FAILED,
                                       "select_money_pot");
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    return TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_NOT_FOUND,
      TALER_EC_MERCHANT_GENERIC_MONEY_POT_UNKNOWN,
      pot_id_str);
  }

  {
    MHD_RESULT ret;

    ret = TALER_MHD_REPLY_JSON_PACK (
      connection,
      MHD_HTTP_OK,
      GNUNET_JSON_pack_string ("description",
                               description),
      GNUNET_JSON_pack_string ("pot_name",
                               pot_name),
      (0 == pot_total_len)
      ? GNUNET_JSON_pack_array_steal ("pot_totals",
                                      json_array ())
      : TALER_JSON_pack_amount_array ("pot_totals",
                                      pot_total_len,
                                      pot_totals));
    GNUNET_free (pot_totals);
    GNUNET_free (pot_name);
    GNUNET_free (description);
    return ret;
  }
}
