/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-post-reports.c
 * @brief implementation of POST /private/reports
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-post-reports.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_dbevents.h>

MHD_RESULT
TMH_private_post_reports (const struct TMH_RequestHandler *rh,
                          struct MHD_Connection *connection,
                          struct TMH_HandlerContext *hc)
{
  const char *description;
  const char *program_section;
  const char *mime_type;
  const char *data_source;
  const char *target_address;
  struct GNUNET_TIME_Relative frequency;
  struct GNUNET_TIME_Relative frequency_shift
    = GNUNET_TIME_UNIT_ZERO;
  enum GNUNET_DB_QueryStatus qs;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("description",
                             &description),
    GNUNET_JSON_spec_string ("program_section",
                             &program_section),
    GNUNET_JSON_spec_string ("mime_type",
                             &mime_type),
    GNUNET_JSON_spec_string ("data_source",
                             &data_source),
    GNUNET_JSON_spec_string ("target_address",
                             &target_address),
    GNUNET_JSON_spec_relative_time ("report_frequency",
                                    &frequency),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_relative_time ("report_frequency_shift",
                                      &frequency_shift),
      NULL),
    GNUNET_JSON_spec_end ()
  };
  uint64_t report_id;

  (void) rh;

  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
    {
      GNUNET_break_op (0);
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
    }
  }
  {
    char *section;

    /* Check program_section exists in config! */
    GNUNET_asprintf (&section,
                     "report-generator-%s",
                     program_section);
    if (GNUNET_YES !=
        GNUNET_CONFIGURATION_have_value (TMH_cfg,
                                         section,
                                         "BINARY"))
    {
      GNUNET_free (section);
      GNUNET_break (0);
      return TALER_MHD_reply_with_error (
        connection,
        MHD_HTTP_NOT_IMPLEMENTED,
        TALER_EC_MERCHANT_GENERIC_REPORT_GENERATOR_UNCONFIGURED,
        program_section);
    }
    GNUNET_free (section);
  }
  qs = TMH_db->insert_report (TMH_db->cls,
                              hc->instance->settings.id,
                              program_section,
                              description,
                              mime_type,
                              data_source,
                              target_address,
                              frequency,
                              frequency_shift,
                              &report_id);
  if (qs < 0)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_STORE_FAILED,
                                       "insert_report");
  }

  /* FIXME-Optimization: do trigger inside of transaction above... */
  {
    struct GNUNET_DB_EventHeaderP ev = {
      .size = htons (sizeof (ev)),
      .type = htons (TALER_DBEVENT_MERCHANT_REPORT_UPDATE)
    };

    TMH_db->event_notify (TMH_db->cls,
                          &ev,
                          NULL,
                          0);
  }

  return TALER_MHD_REPLY_JSON_PACK (
    connection,
    MHD_HTTP_OK,
    GNUNET_JSON_pack_uint64 ("report_serial_id",
                             report_id));
}
