/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Logger,
  PreparePayResultType,
  succeedOrThrow,
  TalerErrorCode,
  TalerMerchantInstanceHttpClient,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const logger = new Logger("test-denom-unoffered");

export async function runDenomUnofferedTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  // Make the exchange forget the denomination.
  // Effectively we completely reset the exchange,
  // but keep the exchange master public key.

  await merchant.stop();

  await exchange.stop();
  await exchange.purgeDatabase();
  await exchange.purgeSecmodKeys();
  await exchange.start();
  await exchange.pingUntilAvailable();

  await merchant.start();
  await merchant.pingUntilAvailable();

  t.logStep("services restarted");

  const orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:5",
        fulfillment_url: "taler://fulfillment-success/thx",
      },
    }),
  );

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  t.logStep("order prepared");

  const confirmResp = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  t.logStep("order confirmed");

  // Currently, the payment transaction will be aborted.
  // In the future, the wallet should also do coin re-selection
  // when a denomination is lost.  But we might want to ask the user
  // first, since they lost money.
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const tx = await walletClient.call(WalletApiOperation.GetTransactionById, {
    transactionId: confirmResp.transactionId,
  });

  t.assertDeepEqual(
    (tx.abortReason?.errorDetails as any)?.code,
    TalerErrorCode.MERCHANT_GENERIC_EXCHANGE_UNEXPECTED_STATUS,
  );

  // Force updating the exchange entry so that the wallet knows about the new denominations.
  await walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
    exchangeBaseUrl: exchange.baseUrl,
    force: true,
  });

  logger.info(`withdrawing (with new denoms)`);

  // Now withdrawal should work again.
  const wres2 = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres2.withdrawalFinishedCond;

  const txs = await walletClient.call(WalletApiOperation.GetTransactions, {
    sort: "stable-ascending",
  });
  console.log(JSON.stringify(txs, undefined, 2));

  const txTypes = txs.transactions.map((x) => x.type);

  t.assertDeepEqual(txTypes, [
    "withdrawal",
    "payment",
    "denom-loss",
    "withdrawal",
  ]);
}

runDenomUnofferedTest.suites = ["wallet"];
