/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
  details.

  You should have received a copy of the GNU Affero General Public License
  along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant/backend/taler-merchant-httpd_private-patch-report-ID.c
 * @brief implementation of PATCH /private/reports/$REPORT_ID
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-patch-report-ID.h"
#include <taler/taler_json_lib.h>
#include <taler/taler_dbevents.h>

MHD_RESULT
TMH_private_patch_report (const struct TMH_RequestHandler *rh,
                          struct MHD_Connection *connection,
                          struct TMH_HandlerContext *hc)
{
  const char *report_id_str = hc->infix;
  unsigned long long report_id;
  const char *description;
  const char *program_section;
  const char *mime_type;
  const char *data_source;
  const char *target_address;
  struct GNUNET_TIME_Relative frequency;
  struct GNUNET_TIME_Relative frequency_shift;
  enum GNUNET_DB_QueryStatus qs;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("description",
                             &description),
    GNUNET_JSON_spec_string ("program_section",
                             &program_section),
    GNUNET_JSON_spec_string ("mime_type",
                             &mime_type),
    GNUNET_JSON_spec_string ("data_source",
                             &data_source),
    GNUNET_JSON_spec_string ("target_address",
                             &target_address),
    GNUNET_JSON_spec_relative_time ("report_frequency",
                                    &frequency),
    GNUNET_JSON_spec_relative_time ("report_frequency_shift",
                                    &frequency_shift),
    GNUNET_JSON_spec_end ()
  };

  (void) rh;
  {
    char dummy;

    if (1 != sscanf (report_id_str,
                     "%llu%c",
                     &report_id,
                     &dummy))
    {
      GNUNET_break_op (0);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "report_id");
    }
  }

  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
    {
      GNUNET_break_op (0);
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
    }
  }

  qs = TMH_db->update_report (TMH_db->cls,
                              hc->instance->settings.id,
                              report_id,
                              program_section,
                              description,
                              mime_type,
                              data_source,
                              target_address,
                              frequency,
                              frequency_shift);
  if (qs < 0)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_DB_STORE_FAILED,
                                       "update_report");
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_NOT_FOUND,
                                       TALER_EC_MERCHANT_GENERIC_REPORT_UNKNOWN,
                                       report_id_str);
  }

  /* FIXME-Optimization: Trigger MERCHANT_REPORT_UPDATE event inside of UPDATE transaction */
  {
    struct GNUNET_DB_EventHeaderP ev = {
      .size = htons (sizeof (ev)),
      .type = htons (TALER_DBEVENT_MERCHANT_REPORT_UPDATE)
    };

    TMH_db->event_notify (TMH_db->cls,
                          &ev,
                          NULL,
                          0);
  }

  return TALER_MHD_reply_static (connection,
                                 MHD_HTTP_NO_CONTENT,
                                 NULL,
                                 NULL,
                                 0);
}
