/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_update_money_pot.c
 * @brief Implementation of the update_money_pot function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_update_money_pot.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_update_money_pot (
  void *cls,
  const char *instance_id,
  uint64_t money_pot_id,
  const char *name,
  const char *description,
  size_t opt_len,
  const struct TALER_Amount *old_pot_totals,
  size_t npt_len,
  const struct TALER_Amount *new_pot_totals,
  bool *conflict_total,
  bool *conflict_name)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_uint64 (&money_pot_id),
    GNUNET_PQ_query_param_string (name),
    GNUNET_PQ_query_param_string (description),
    (NULL == old_pot_totals)
    ? GNUNET_PQ_query_param_null ()
    : TALER_PQ_query_param_array_amount_with_currency (opt_len,
                                                       old_pot_totals,
                                                       pg->conn),
    (NULL == new_pot_totals)
    ? GNUNET_PQ_query_param_null ()
    : TALER_PQ_query_param_array_amount_with_currency (npt_len,
                                                       new_pot_totals,
                                                       pg->conn),
    GNUNET_PQ_query_param_end
  };
  bool not_found;
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_bool ("conflict_total",
                                conflict_total),
    GNUNET_PQ_result_spec_bool ("conflict_name",
                                conflict_name),
    GNUNET_PQ_result_spec_bool ("not_found",
                                &not_found),
    GNUNET_PQ_result_spec_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "update_money_pot",
           "SELECT"
           " out_conflict_name AS conflict_name"
           ",out_conflict_total AS conflict_total"
           ",out_not_found AS not_found"
           " FROM merchant_do_update_money_pot"
           "($1,$2,$3,$4,$5,$6);");
  qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                 "update_money_pot",
                                                 params,
                                                 rs);
  GNUNET_PQ_cleanup_query_params_closures (params);
  if (qs <= 0)
    return qs;
  if (not_found)
    return GNUNET_DB_STATUS_SUCCESS_NO_RESULTS;
  return GNUNET_DB_STATUS_SUCCESS_ONE_RESULT;
}
