/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { PaytoUri, stringifyPaytoUri } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { Loading, useTranslationContext } from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { ErrorAlertView } from "../components/CurrentAlerts.js";
import { alertFromError } from "../context/alert.js";
import { useBackendContext } from "../context/backend.js";
import { useAsyncAsHook } from "../hooks/useAsyncAsHook.js";

interface Props {
  account: PaytoUri;
}

export function SupportedBanksForAccount({ account }: Props): VNode {
  const api = useBackendContext();
  const { i18n } = useTranslationContext();
  const state = useAsyncAsHook(() => {
    return api.wallet.call(WalletApiOperation.GetBankingChoicesForPayto, {
      paytoUri: stringifyPaytoUri(account),
    });
  });
  if (!state) {
    return <Loading />;
  }

  if (state.hasError) {
    return (
      <ErrorAlertView
        error={alertFromError(
          i18n,
          i18n.str`Could not bank choices for account`,
          state,
        )}
      />
    );
  }

  return (
    <div>
      {state.response.choices.map((ch) => {
        return <a href={ch.uri}>{ch.label}</a>;
      })}
    </div>
  );
}
