.\" Copyright (c) 2024 Andrea Biscuola <a@abiscuola.com>
.\"
.\" Permission to use, copy, modify, and distribute this software for any
.\" purpose with or without fee is hereby granted, provided that the above
.\" copyright notice and this permission notice appear in all copies.
.\"
.\" THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
.\" WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
.\" MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
.\" ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
.\" WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
.\" ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
.\" OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
.\"
.Dd May 20, 2024
.Dt IRCTK-EXT-SDK n
.Os
.Sh NAME
.Nm irctk-ext-sdk
.Nd tcl software development kit for irctk extensions
.Sh SYNOPSIS
.Cm ::irctk::filter Ar type Op Ar callback
.Pp
.Cm ::irctk::init Ar name Ar version Ar capabilities Op Ar callback
.Pp
.Cm ::irctk::ircsend Ar net Ar chan Ar cmd Ar line
.Pp
.Cm ::irctk::loop
.Sh DESCRIPTION
The IRCTk extensions SDK, is a tcl package designed to simplify the creation
of IRCTk extensions in TCL.
.Pp
The subsystem calls any
.Ar callback
defined in the extension with a message as an argument to it.
A call to a callback have the following form:
.Bd -literal -offset Ds
callback msg
.Ed
.Pp
Messages passed to a callback are tcl dictionaries with different fields based
on the type.
The
.Dq irc
message type has the following fields:
.Bl -tag -width Ds
.It id
Message Id.
.It type
Message type.
.It timestamp
Timestamp of the message, in seconds, from the epoch.
.It cid
The id of the channel in IRCTk.
.It nick
The nick of the user that sent the message
.It level
Message level, from
.Dq info ,
to
.Dq mention.
.It focus
true or false.
Define if the application window is on focus.
.It status
Client user's status.
Might be
.Dq away
or not.
.It network
The network name.
.It channel
The channel name.
.It tags
A list of IRCv3 tags
.It command
The IRC command.
.It args
The command arguments.
.El
.Pp
Fields for the
.Dq plumb
message are:
.Bl -tag -width Ds
.It id
Message Id (EMPTY).
.It type
The message type.
.It cid
The Id of the channel where the plumb happened.
.It network
The network name.
.It channel
The channel name.
.It data
The text string.
.El
.Pp
.Cm ::irctk::init
initializes the subsystem and perform the initial handshake with IRCTk using
.Ar name
and
.Ar version
as the extension's parameters.
The
.Ar capabilities
argument is a list of IRCv3 capabilities the extension is able to manage, that
IRCTk will ask the server to enable.
The optional
.Ar callback ,
if provided, will handle all the messages routed to the extension without a
proper filter in place.
.Pp
.Cm ::irctk::filter
defines a new policy for IRCTk to follow, related to the extension.
In particular, it informs IRCTk to route only specific message types,
or specific message commands, to the extension itself.
The optional
.Ar callback
will be called, by the sdk, whenever a message of the specified
.Ar type
is received.
The possible message types are:
.Bl -bullet -width Ds
.It
plumb
.It
irc
.El
.Pp
As for the type irc, it's also possible to specify filters for specific
commands, by first issuing a filter request for the irc type, and
then by issuing a request for the commands themselves:
.Bd -literal -offset Ds
::irctk::filter irc
::irctk::filter JOIN "callback"
::irctk::filter PRIVMSG "callback"
.Ed
.Pp
Issuing a request only for the irc type, like the following:
.Bd -literal -offset Ds
::irctk::filter irc "callback"
.Ed
.Pp
Will cause IRCTk to route all the messages of type irc to the extension.
.Pp
The
.Cm ::irc::loop
command starts the main extension loop.
In essence, it starts the tcl event loop, processing incoming messages, and
routing them to the appropriate callback for the required filters, or using the
default handler if the message type doesn't have a filter specified.
.Pp
It's possible, of course, for an extension to add it's own events to the event
loop, as the sdk just takes care of starting the standard tcl one.
An example may be an extension that open a socket, binding a read event
to get some data from it.
.Pp
The
.Cm ::irctk::ircsend
command, will send a message of type IRC to IRCTk, with
.Ar net
as a network of destination,
.Ar chan
as the specific channel,
.Ar cmd
as the IRC command and
.Ar line
as a string for the given command.
.Pp
Sending an irc message to a channel that doesn't exists in the specified
network, will cause IRCTk to create a, so called,
.Dq read only
channel, where the body of the message will be visualized, but without any
possibility for the user to interact with the channel itself.
.Pp
This is useful, for example, to create extensions that doesn't need to send
messages to an IRC server, but only show information to the client's user.
.Sh ENVIRONMENT
.Bl -tag -width Ds
.It Ev IRCTK_TCL_SDK
Path to the SDk source file.
.It Ev IRCTK_LOGS_PATH
Path to the IRCTk chat logs directory.
.El
.Sh EXAMPLES
The following is a, basic,
.Dq Hello, world!
Extension, showing the general layout:
.Bd -literal -offset Ds
#!/usr/bin/env tclsh8.6

source $::env(IRCTK_TCL_SDK)

set name "Hello"
set version 1.0.0

proc helloworld {msg} {
	set network [dict get $msg network]
	set channel [dict get $msg channel]

	::irctk::ircsend $network $channel PRIVMSG "Hello world!"
}

::irctk::init $name $version {}
::irctk::filter irc {}
::irctk::filter hello helloworld

::irctk::loop
.Ed
.Sh SEE ALSO
.Xr irctk-extensions 7
.Sh AUTHORS
.An Andrea Biscuola Aq Mt a@abiscuola.com
