"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSearchRelevanceRoutes = registerSearchRelevanceRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerSearchRelevanceRoutes(router) {
  router.post({
    path: _common.ServiceEndpoints.QuerySets,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string(),
        sampling: _configSchema.schema.string(),
        querySetSize: _configSchema.schema.number()
      })
    }
  }, backendAction('POST', _common.BackendEndpoints.QuerySets));
  router.put({
    path: _common.ServiceEndpoints.QuerySets,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string(),
        sampling: _configSchema.schema.string(),
        querySetQueries: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.object({
          queryText: _configSchema.schema.string(),
          referenceAnswer: _configSchema.schema.maybe(_configSchema.schema.string())
        }), {
          minSize: 1
        }), _configSchema.schema.string()])
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.QuerySets));
  router.get({
    path: _common.ServiceEndpoints.QuerySets,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.QuerySets));
  router.delete({
    path: `${_common.ServiceEndpoints.QuerySets}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.QuerySets));
  router.put({
    path: _common.ServiceEndpoints.SearchConfigurations,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        index: _configSchema.schema.string(),
        query: _configSchema.schema.string(),
        searchPipeline: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.SearchConfigurations));
  router.get({
    path: _common.ServiceEndpoints.SearchConfigurations,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.SearchConfigurations));
  router.delete({
    path: `${_common.ServiceEndpoints.SearchConfigurations}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.SearchConfigurations));
  router.post({
    path: _common.ServiceEndpoints.Experiments,
    validate: {
      body: _configSchema.schema.object({
        querySetId: _configSchema.schema.string(),
        searchConfigurationList: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        size: _configSchema.schema.number(),
        type: _configSchema.schema.string(),
        // TODO: make mandatory conditional on experiment type
        judgmentList: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.Experiments));
  router.get({
    path: _common.ServiceEndpoints.Experiments,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.Experiments));
  router.get({
    path: `${_common.ServiceEndpoints.Experiments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.Experiments));
  router.get({
    path: `${_common.ServiceEndpoints.SearchConfigurations}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.SearchConfigurations));
  router.get({
    path: `${_common.ServiceEndpoints.QuerySets}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.QuerySets));
  router.delete({
    path: `${_common.ServiceEndpoints.Experiments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.Experiments));
  router.put({
    path: _common.ServiceEndpoints.Judgments,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        type: _configSchema.schema.string(),
        querySetId: _configSchema.schema.maybe(_configSchema.schema.string()),
        searchConfigurationList: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        size: _configSchema.schema.maybe(_configSchema.schema.number()),
        modelId: _configSchema.schema.maybe(_configSchema.schema.string()),
        tokenLimit: _configSchema.schema.maybe(_configSchema.schema.string()),
        ignoreFailure: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        contextFields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        clickModel: _configSchema.schema.maybe(_configSchema.schema.string()),
        maxRank: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.Judgments));
  router.get({
    path: _common.ServiceEndpoints.Judgments,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.Judgments));
  router.get({
    path: `${_common.ServiceEndpoints.Judgments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.Judgments));
  router.delete({
    path: `${_common.ServiceEndpoints.Judgments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.Judgments));
}
const backendAction = (method, path) => {
  return async (context, req, res) => {
    const dataSourceId = req.query.data_source;
    const caller = dataSourceId ? context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI : context.core.opensearch.legacy.client.callAsCurrentUser;
    try {
      let response;
      if (method === 'DELETE') {
        const {
          id
        } = req.params;
        const deletePath = `${path}/${id}`;
        response = await caller('transport.request', {
          method,
          path: deletePath
        });
      } else if (method === 'GET' && req.params.id) {
        // Handle GET request for individual experiment
        const getPath = `${path}/${req.params.id}`;
        response = await caller('transport.request', {
          method,
          path: getPath
        });
      } else {
        // Handle PUT, POST, GET as before
        response = await caller('transport.request', {
          method,
          path,
          ...(method === 'POST' || method === 'PUT' ? {
            body: req.body
          } : {})
        });
      }
      return res.ok({
        body: response
      });
    } catch (err) {
      var _err$body;
      console.error('Failed to call search-relevance APIs', err);
      return res.customError({
        statusCode: err.statusCode || 500,
        body: {
          message: err.message,
          attributes: {
            error: ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.error) || err.message
          }
        }
      });
    }
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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