"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@osd/config-schema");
var _constants = require("../../common/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function defineRoutes(router, dataSourceEnabled) {
  router.get({
    path: '/api/top_queries',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query;
      if (!dataSourceEnabled || !((_request$query = request.query) !== null && _request$query !== void 0 && _request$query.dataSourceId)) {
        const client = context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser;
        const res = await client('queryInsights.getTopNQueries');
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } else {
        var _request$query2;
        const client = context.dataSource.opensearch.legacy.getClient((_request$query2 = request.query) === null || _request$query2 === void 0 ? void 0 : _request$query2.dataSourceId);
        const res = await client.callAPI('queryInsights.getTopNQueries', {});
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      }
    } catch (error) {
      console.error('Unable to get top queries: ', error);
      return response.ok({
        body: {
          ok: false,
          response: error.message
        }
      });
    }
  });
  router.get({
    path: '/api/top_queries/latency',
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        to: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        id: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
        verbose: _configSchema.schema.maybe(_configSchema.schema.boolean({
          defaultValue: false
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query3;
      const {
        from,
        to,
        id,
        verbose
      } = request.query;
      const params = {
        from,
        to,
        id,
        verbose
      };
      if (!dataSourceEnabled || !((_request$query3 = request.query) !== null && _request$query3 !== void 0 && _request$query3.dataSourceId)) {
        const client = context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser;
        const res = id != null ? await client('queryInsights.getTopNQueriesLatencyForId', params) : await client('queryInsights.getTopNQueriesLatency', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } else {
        var _request$query4;
        const client = context.dataSource.opensearch.legacy.getClient((_request$query4 = request.query) === null || _request$query4 === void 0 ? void 0 : _request$query4.dataSourceId);
        const res = id != null ? await client.callAPI('queryInsights.getTopNQueriesLatencyForId', params) : await client.callAPI('queryInsights.getTopNQueriesLatency', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      }
    } catch (error) {
      console.error('Unable to get top queries (latency): ', error);
      return response.ok({
        body: {
          ok: false,
          response: error.message
        }
      });
    }
  });
  router.get({
    path: '/api/top_queries/cpu',
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        to: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        id: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
        verbose: _configSchema.schema.maybe(_configSchema.schema.boolean({
          defaultValue: false
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query5;
      const {
        from,
        to,
        id,
        verbose
      } = request.query;
      const params = {
        from,
        to,
        id,
        verbose
      };
      if (!dataSourceEnabled || !((_request$query5 = request.query) !== null && _request$query5 !== void 0 && _request$query5.dataSourceId)) {
        const client = context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser;
        const res = id != null ? await client('queryInsights.getTopNQueriesCpuForId', params) : await client('queryInsights.getTopNQueriesCpu', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } else {
        var _request$query6;
        const client = context.dataSource.opensearch.legacy.getClient((_request$query6 = request.query) === null || _request$query6 === void 0 ? void 0 : _request$query6.dataSourceId);
        const res = id != null ? await client.callAPI('queryInsights.getTopNQueriesCpuForId', params) : await client.callAPI('queryInsights.getTopNQueriesCpu', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      }
    } catch (error) {
      console.error('Unable to get top queries (cpu): ', error);
      return response.ok({
        body: {
          ok: false,
          response: error.message
        }
      });
    }
  });
  router.get({
    path: '/api/top_queries/memory',
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        to: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        id: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
        verbose: _configSchema.schema.maybe(_configSchema.schema.boolean({
          defaultValue: false
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query7;
      const {
        from,
        to,
        id,
        verbose
      } = request.query;
      const params = {
        from,
        to,
        id,
        verbose
      };
      if (!dataSourceEnabled || !((_request$query7 = request.query) !== null && _request$query7 !== void 0 && _request$query7.dataSourceId)) {
        const client = context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser;
        const res = id != null ? await client('queryInsights.getTopNQueriesMemoryForId', params) : await client('queryInsights.getTopNQueriesMemory', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } else {
        var _request$query8;
        const client = context.dataSource.opensearch.legacy.getClient((_request$query8 = request.query) === null || _request$query8 === void 0 ? void 0 : _request$query8.dataSourceId);
        const res = id != null ? await client.callAPI('queryInsights.getTopNQueriesMemoryForId', params) : await client.callAPI('queryInsights.getTopNQueriesMemory', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      }
    } catch (error) {
      console.error('Unable to get top queries (memory): ', error);
      return response.ok({
        body: {
          ok: false,
          response: error.message
        }
      });
    }
  });
  router.get({
    path: '/api/settings',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query9;
      if (!dataSourceEnabled || !((_request$query9 = request.query) !== null && _request$query9 !== void 0 && _request$query9.dataSourceId)) {
        const client = context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser;
        const res = await client('queryInsights.getSettings');
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } else {
        var _request$query10;
        const client = context.dataSource.opensearch.legacy.getClient((_request$query10 = request.query) === null || _request$query10 === void 0 ? void 0 : _request$query10.dataSourceId);
        const res = await client.callAPI('queryInsights.getSettings', {});
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      }
    } catch (error) {
      console.error('Unable to get top queries: ', error);
      return response.ok({
        body: {
          ok: false,
          response: error.message
        }
      });
    }
  });
  router.put({
    path: '/api/update_settings',
    validate: {
      query: _configSchema.schema.object({
        metric: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        enabled: _configSchema.schema.maybe(_configSchema.schema.boolean({
          defaultValue: false
        })),
        top_n_size: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        window_size: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        exporterType: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        group_by: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
        delete_after_days: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query11;
      const query = request.query;
      const params = {
        body: {
          persistent: {
            [`search.insights.top_queries.${query.metric}.enabled`]: query.enabled,
            [`search.insights.top_queries.${query.metric}.top_n_size`]: query.top_n_size,
            [`search.insights.top_queries.${query.metric}.window_size`]: query.window_size
          }
        }
      };
      if (query.group_by !== '') {
        params.body.persistent['search.insights.top_queries.grouping.group_by'] = query.group_by;
      }
      if (query.delete_after_days !== '') {
        params.body.persistent['search.insights.top_queries.exporter.delete_after_days'] = query.delete_after_days;
      }
      if (query.exporterType !== '') {
        params.body.persistent['search.insights.top_queries.exporter.type'] = query.exporterType === _constants.EXPORTER_TYPE.localIndex ? query.exporterType : _constants.EXPORTER_TYPE.none;
      }
      if (!dataSourceEnabled || !((_request$query11 = request.query) !== null && _request$query11 !== void 0 && _request$query11.dataSourceId)) {
        const client = context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser;
        const res = await client('queryInsights.setSettings', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      } else {
        var _request$query12;
        const client = context.dataSource.opensearch.legacy.getClient((_request$query12 = request.query) === null || _request$query12 === void 0 ? void 0 : _request$query12.dataSourceId);
        const res = await client.callAPI('queryInsights.setSettings', params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: res
          }
        });
      }
    } catch (error) {
      console.error('Unable to set settings: ', error);
      return response.ok({
        body: {
          ok: false,
          response: error.message
        }
      });
    }
  });
  router.get({
    path: '/api/live_queries',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query13;
      const client = !dataSourceEnabled || !((_request$query13 = request.query) !== null && _request$query13 !== void 0 && _request$query13.dataSourceId) ? context.queryInsights_plugin.queryInsightsClient.asScoped(request).callAsCurrentUser : context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
      const res = await client('queryInsights.getLiveQueries');
      if (!res || res.ok === false) {
        throw new Error((res === null || res === void 0 ? void 0 : res.error) || 'Query Insights service returned an error');
      }
      return response.ok({
        body: {
          ok: true,
          response: res
        }
      });
    } catch (error) {
      var _error$statusCode;
      console.error('Unable to get live queries: ', error);
      return response.customError({
        statusCode: (_error$statusCode = error.statusCode) !== null && _error$statusCode !== void 0 ? _error$statusCode : 500,
        body: {
          message: error.message || 'Internal server error'
        }
      });
    }
  });
  router.post({
    path: '/api/tasks/{taskId}/cancel',
    validate: {
      params: _configSchema.schema.object({
        taskId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const esClient = context.core.opensearch.client.asCurrentUser;
      const {
        taskId
      } = request.params;
      const res = await esClient.transport.request({
        method: 'POST',
        path: `/_tasks/${taskId}/_cancel`
      });
      if (!res) {
        throw new Error('failed');
      }
      return response.ok({
        body: {
          ok: true,
          res
        }
      });
    } catch (error) {
      var _error$statusCode2;
      console.error(error);
      return response.customError({
        statusCode: (_error$statusCode2 = error.statusCode) !== null && _error$statusCode2 !== void 0 ? _error$statusCode2 : 500,
        body: {
          message: error.message || 'Internal server error'
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9jb25zdGFudHMiLCJkZWZpbmVSb3V0ZXMiLCJyb3V0ZXIiLCJkYXRhU291cmNlRW5hYmxlZCIsImdldCIsInBhdGgiLCJ2YWxpZGF0ZSIsInF1ZXJ5Iiwic2NoZW1hIiwib2JqZWN0IiwiZGF0YVNvdXJjZUlkIiwibWF5YmUiLCJzdHJpbmciLCJjb250ZXh0IiwicmVxdWVzdCIsInJlc3BvbnNlIiwiX3JlcXVlc3QkcXVlcnkiLCJjbGllbnQiLCJxdWVyeUluc2lnaHRzX3BsdWdpbiIsInF1ZXJ5SW5zaWdodHNDbGllbnQiLCJhc1Njb3BlZCIsImNhbGxBc0N1cnJlbnRVc2VyIiwicmVzIiwiY3VzdG9tIiwic3RhdHVzQ29kZSIsImJvZHkiLCJvayIsIl9yZXF1ZXN0JHF1ZXJ5MiIsImRhdGFTb3VyY2UiLCJvcGVuc2VhcmNoIiwibGVnYWN5IiwiZ2V0Q2xpZW50IiwiY2FsbEFQSSIsImVycm9yIiwiY29uc29sZSIsIm1lc3NhZ2UiLCJmcm9tIiwiZGVmYXVsdFZhbHVlIiwidG8iLCJpZCIsInZlcmJvc2UiLCJib29sZWFuIiwiX3JlcXVlc3QkcXVlcnkzIiwicGFyYW1zIiwiX3JlcXVlc3QkcXVlcnk0IiwiX3JlcXVlc3QkcXVlcnk1IiwiX3JlcXVlc3QkcXVlcnk2IiwiX3JlcXVlc3QkcXVlcnk3IiwiX3JlcXVlc3QkcXVlcnk4IiwiX3JlcXVlc3QkcXVlcnk5IiwiX3JlcXVlc3QkcXVlcnkxMCIsInB1dCIsIm1ldHJpYyIsImVuYWJsZWQiLCJ0b3Bfbl9zaXplIiwid2luZG93X3NpemUiLCJleHBvcnRlclR5cGUiLCJncm91cF9ieSIsImRlbGV0ZV9hZnRlcl9kYXlzIiwiX3JlcXVlc3QkcXVlcnkxMSIsInBlcnNpc3RlbnQiLCJFWFBPUlRFUl9UWVBFIiwibG9jYWxJbmRleCIsIm5vbmUiLCJfcmVxdWVzdCRxdWVyeTEyIiwiX3JlcXVlc3QkcXVlcnkxMyIsIkVycm9yIiwiX2Vycm9yJHN0YXR1c0NvZGUiLCJjdXN0b21FcnJvciIsInBvc3QiLCJ0YXNrSWQiLCJlc0NsaWVudCIsImNvcmUiLCJhc0N1cnJlbnRVc2VyIiwidHJhbnNwb3J0IiwibWV0aG9kIiwiX2Vycm9yJHN0YXR1c0NvZGUyIl0sInNvdXJjZXMiOlsiaW5kZXgudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgeyBzY2hlbWEgfSBmcm9tICdAb3NkL2NvbmZpZy1zY2hlbWEnO1xuaW1wb3J0IHsgSVJvdXRlciB9IGZyb20gJy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQgeyBFWFBPUlRFUl9UWVBFIH0gZnJvbSAnLi4vLi4vY29tbW9uL2NvbnN0YW50cyc7XG5cbmV4cG9ydCBmdW5jdGlvbiBkZWZpbmVSb3V0ZXMocm91dGVyOiBJUm91dGVyLCBkYXRhU291cmNlRW5hYmxlZDogYm9vbGVhbikge1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL3RvcF9xdWVyaWVzJyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHF1ZXJ5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBkYXRhU291cmNlSWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpID0+IHtcbiAgICAgIHRyeSB7XG4gICAgICAgIGlmICghZGF0YVNvdXJjZUVuYWJsZWQgfHwgIXJlcXVlc3QucXVlcnk/LmRhdGFTb3VyY2VJZCkge1xuICAgICAgICAgIGNvbnN0IGNsaWVudCA9IGNvbnRleHQucXVlcnlJbnNpZ2h0c19wbHVnaW4ucXVlcnlJbnNpZ2h0c0NsaWVudC5hc1Njb3BlZChyZXF1ZXN0KVxuICAgICAgICAgICAgLmNhbGxBc0N1cnJlbnRVc2VyO1xuICAgICAgICAgIGNvbnN0IHJlcyA9IGF3YWl0IGNsaWVudCgncXVlcnlJbnNpZ2h0cy5nZXRUb3BOUXVlcmllcycpO1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICAgICAgcmVzcG9uc2U6IHJlcyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5kYXRhU291cmNlLm9wZW5zZWFyY2gubGVnYWN5LmdldENsaWVudChcbiAgICAgICAgICAgIHJlcXVlc3QucXVlcnk/LmRhdGFTb3VyY2VJZFxuICAgICAgICAgICk7XG4gICAgICAgICAgY29uc3QgcmVzID0gYXdhaXQgY2xpZW50LmNhbGxBUEkoJ3F1ZXJ5SW5zaWdodHMuZ2V0VG9wTlF1ZXJpZXMnLCB7fSk7XG4gICAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgICAgICByZXNwb25zZTogcmVzLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignVW5hYmxlIHRvIGdldCB0b3AgcXVlcmllczogJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICAgIHJlc3BvbnNlOiBlcnJvci5tZXNzYWdlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6ICcvYXBpL3RvcF9xdWVyaWVzL2xhdGVuY3knLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGZyb206IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgdG86IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgaWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgZGF0YVNvdXJjZUlkOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICB2ZXJib3NlOiBzY2hlbWEubWF5YmUoc2NoZW1hLmJvb2xlYW4oeyBkZWZhdWx0VmFsdWU6IGZhbHNlIH0pKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCB7IGZyb20sIHRvLCBpZCwgdmVyYm9zZSB9ID0gcmVxdWVzdC5xdWVyeTtcbiAgICAgICAgY29uc3QgcGFyYW1zID0geyBmcm9tLCB0bywgaWQsIHZlcmJvc2UgfTtcbiAgICAgICAgaWYgKCFkYXRhU291cmNlRW5hYmxlZCB8fCAhcmVxdWVzdC5xdWVyeT8uZGF0YVNvdXJjZUlkKSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5xdWVyeUluc2lnaHRzX3BsdWdpbi5xdWVyeUluc2lnaHRzQ2xpZW50LmFzU2NvcGVkKHJlcXVlc3QpXG4gICAgICAgICAgICAuY2FsbEFzQ3VycmVudFVzZXI7XG4gICAgICAgICAgY29uc3QgcmVzID1cbiAgICAgICAgICAgIGlkICE9IG51bGxcbiAgICAgICAgICAgICAgPyBhd2FpdCBjbGllbnQoJ3F1ZXJ5SW5zaWdodHMuZ2V0VG9wTlF1ZXJpZXNMYXRlbmN5Rm9ySWQnLCBwYXJhbXMpXG4gICAgICAgICAgICAgIDogYXdhaXQgY2xpZW50KCdxdWVyeUluc2lnaHRzLmdldFRvcE5RdWVyaWVzTGF0ZW5jeScsIHBhcmFtcyk7XG4gICAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgICAgICByZXNwb25zZTogcmVzLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmRhdGFTb3VyY2Uub3BlbnNlYXJjaC5sZWdhY3kuZ2V0Q2xpZW50KFxuICAgICAgICAgICAgcmVxdWVzdC5xdWVyeT8uZGF0YVNvdXJjZUlkXG4gICAgICAgICAgKTtcbiAgICAgICAgICBjb25zdCByZXMgPVxuICAgICAgICAgICAgaWQgIT0gbnVsbFxuICAgICAgICAgICAgICA/IGF3YWl0IGNsaWVudC5jYWxsQVBJKCdxdWVyeUluc2lnaHRzLmdldFRvcE5RdWVyaWVzTGF0ZW5jeUZvcklkJywgcGFyYW1zKVxuICAgICAgICAgICAgICA6IGF3YWl0IGNsaWVudC5jYWxsQVBJKCdxdWVyeUluc2lnaHRzLmdldFRvcE5RdWVyaWVzTGF0ZW5jeScsIHBhcmFtcyk7XG4gICAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgICAgICByZXNwb25zZTogcmVzLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignVW5hYmxlIHRvIGdldCB0b3AgcXVlcmllcyAobGF0ZW5jeSk6ICcsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgICByZXNwb25zZTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS90b3BfcXVlcmllcy9jcHUnLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGZyb206IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgdG86IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgaWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgZGF0YVNvdXJjZUlkOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICB2ZXJib3NlOiBzY2hlbWEubWF5YmUoc2NoZW1hLmJvb2xlYW4oeyBkZWZhdWx0VmFsdWU6IGZhbHNlIH0pKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCB7IGZyb20sIHRvLCBpZCwgdmVyYm9zZSB9ID0gcmVxdWVzdC5xdWVyeTtcbiAgICAgICAgY29uc3QgcGFyYW1zID0geyBmcm9tLCB0bywgaWQsIHZlcmJvc2UgfTtcblxuICAgICAgICBpZiAoIWRhdGFTb3VyY2VFbmFibGVkIHx8ICFyZXF1ZXN0LnF1ZXJ5Py5kYXRhU291cmNlSWQpIHtcbiAgICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LnF1ZXJ5SW5zaWdodHNfcGx1Z2luLnF1ZXJ5SW5zaWdodHNDbGllbnQuYXNTY29wZWQocmVxdWVzdClcbiAgICAgICAgICAgIC5jYWxsQXNDdXJyZW50VXNlcjtcbiAgICAgICAgICBjb25zdCByZXMgPVxuICAgICAgICAgICAgaWQgIT0gbnVsbFxuICAgICAgICAgICAgICA/IGF3YWl0IGNsaWVudCgncXVlcnlJbnNpZ2h0cy5nZXRUb3BOUXVlcmllc0NwdUZvcklkJywgcGFyYW1zKVxuICAgICAgICAgICAgICA6IGF3YWl0IGNsaWVudCgncXVlcnlJbnNpZ2h0cy5nZXRUb3BOUXVlcmllc0NwdScsIHBhcmFtcyk7XG4gICAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgICAgICByZXNwb25zZTogcmVzLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmRhdGFTb3VyY2Uub3BlbnNlYXJjaC5sZWdhY3kuZ2V0Q2xpZW50KFxuICAgICAgICAgICAgcmVxdWVzdC5xdWVyeT8uZGF0YVNvdXJjZUlkXG4gICAgICAgICAgKTtcbiAgICAgICAgICBjb25zdCByZXMgPVxuICAgICAgICAgICAgaWQgIT0gbnVsbFxuICAgICAgICAgICAgICA/IGF3YWl0IGNsaWVudC5jYWxsQVBJKCdxdWVyeUluc2lnaHRzLmdldFRvcE5RdWVyaWVzQ3B1Rm9ySWQnLCBwYXJhbXMpXG4gICAgICAgICAgICAgIDogYXdhaXQgY2xpZW50LmNhbGxBUEkoJ3F1ZXJ5SW5zaWdodHMuZ2V0VG9wTlF1ZXJpZXNDcHUnLCBwYXJhbXMpO1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICAgICAgcmVzcG9uc2U6IHJlcyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ1VuYWJsZSB0byBnZXQgdG9wIHF1ZXJpZXMgKGNwdSk6ICcsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgICByZXNwb25zZTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS90b3BfcXVlcmllcy9tZW1vcnknLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGZyb206IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgdG86IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgaWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgZGF0YVNvdXJjZUlkOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICB2ZXJib3NlOiBzY2hlbWEubWF5YmUoc2NoZW1hLmJvb2xlYW4oeyBkZWZhdWx0VmFsdWU6IGZhbHNlIH0pKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCB7IGZyb20sIHRvLCBpZCwgdmVyYm9zZSB9ID0gcmVxdWVzdC5xdWVyeTtcbiAgICAgICAgY29uc3QgcGFyYW1zID0geyBmcm9tLCB0bywgaWQsIHZlcmJvc2UgfTtcbiAgICAgICAgaWYgKCFkYXRhU291cmNlRW5hYmxlZCB8fCAhcmVxdWVzdC5xdWVyeT8uZGF0YVNvdXJjZUlkKSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5xdWVyeUluc2lnaHRzX3BsdWdpbi5xdWVyeUluc2lnaHRzQ2xpZW50LmFzU2NvcGVkKHJlcXVlc3QpXG4gICAgICAgICAgICAuY2FsbEFzQ3VycmVudFVzZXI7XG4gICAgICAgICAgY29uc3QgcmVzID1cbiAgICAgICAgICAgIGlkICE9IG51bGxcbiAgICAgICAgICAgICAgPyBhd2FpdCBjbGllbnQoJ3F1ZXJ5SW5zaWdodHMuZ2V0VG9wTlF1ZXJpZXNNZW1vcnlGb3JJZCcsIHBhcmFtcylcbiAgICAgICAgICAgICAgOiBhd2FpdCBjbGllbnQoJ3F1ZXJ5SW5zaWdodHMuZ2V0VG9wTlF1ZXJpZXNNZW1vcnknLCBwYXJhbXMpO1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICAgICAgcmVzcG9uc2U6IHJlcyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5kYXRhU291cmNlLm9wZW5zZWFyY2gubGVnYWN5LmdldENsaWVudChcbiAgICAgICAgICAgIHJlcXVlc3QucXVlcnk/LmRhdGFTb3VyY2VJZFxuICAgICAgICAgICk7XG4gICAgICAgICAgY29uc3QgcmVzID1cbiAgICAgICAgICAgIGlkICE9IG51bGxcbiAgICAgICAgICAgICAgPyBhd2FpdCBjbGllbnQuY2FsbEFQSSgncXVlcnlJbnNpZ2h0cy5nZXRUb3BOUXVlcmllc01lbW9yeUZvcklkJywgcGFyYW1zKVxuICAgICAgICAgICAgICA6IGF3YWl0IGNsaWVudC5jYWxsQVBJKCdxdWVyeUluc2lnaHRzLmdldFRvcE5RdWVyaWVzTWVtb3J5JywgcGFyYW1zKTtcbiAgICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICAgIHN0YXR1c0NvZGU6IDIwMCxcbiAgICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgICAgIHJlc3BvbnNlOiByZXMsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBjb25zb2xlLmVycm9yKCdVbmFibGUgdG8gZ2V0IHRvcCBxdWVyaWVzIChtZW1vcnkpOiAnLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgICAgcmVzcG9uc2U6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvc2V0dGluZ3MnLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGRhdGFTb3VyY2VJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSkgPT4ge1xuICAgICAgdHJ5IHtcbiAgICAgICAgaWYgKCFkYXRhU291cmNlRW5hYmxlZCB8fCAhcmVxdWVzdC5xdWVyeT8uZGF0YVNvdXJjZUlkKSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5xdWVyeUluc2lnaHRzX3BsdWdpbi5xdWVyeUluc2lnaHRzQ2xpZW50LmFzU2NvcGVkKHJlcXVlc3QpXG4gICAgICAgICAgICAuY2FsbEFzQ3VycmVudFVzZXI7XG4gICAgICAgICAgY29uc3QgcmVzID0gYXdhaXQgY2xpZW50KCdxdWVyeUluc2lnaHRzLmdldFNldHRpbmdzJyk7XG4gICAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgICBzdGF0dXNDb2RlOiAyMDAsXG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgICAgICByZXNwb25zZTogcmVzLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmRhdGFTb3VyY2Uub3BlbnNlYXJjaC5sZWdhY3kuZ2V0Q2xpZW50KFxuICAgICAgICAgICAgcmVxdWVzdC5xdWVyeT8uZGF0YVNvdXJjZUlkXG4gICAgICAgICAgKTtcbiAgICAgICAgICBjb25zdCByZXMgPSBhd2FpdCBjbGllbnQuY2FsbEFQSSgncXVlcnlJbnNpZ2h0cy5nZXRTZXR0aW5ncycsIHt9KTtcbiAgICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICAgIHN0YXR1c0NvZGU6IDIwMCxcbiAgICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgICAgIHJlc3BvbnNlOiByZXMsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBjb25zb2xlLmVycm9yKCdVbmFibGUgdG8gZ2V0IHRvcCBxdWVyaWVzOiAnLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgICAgcmVzcG9uc2U6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIHJvdXRlci5wdXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvdXBkYXRlX3NldHRpbmdzJyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHF1ZXJ5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBtZXRyaWM6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgZW5hYmxlZDogc2NoZW1hLm1heWJlKHNjaGVtYS5ib29sZWFuKHsgZGVmYXVsdFZhbHVlOiBmYWxzZSB9KSksXG4gICAgICAgICAgdG9wX25fc2l6ZTogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoeyBkZWZhdWx0VmFsdWU6ICcnIH0pKSxcbiAgICAgICAgICB3aW5kb3dfc2l6ZTogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoeyBkZWZhdWx0VmFsdWU6ICcnIH0pKSxcbiAgICAgICAgICBleHBvcnRlclR5cGU6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgZ3JvdXBfYnk6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgICAgZGF0YVNvdXJjZUlkOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICBkZWxldGVfYWZ0ZXJfZGF5czogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoeyBkZWZhdWx0VmFsdWU6ICcnIH0pKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBxdWVyeSA9IHJlcXVlc3QucXVlcnk7XG4gICAgICAgIGNvbnN0IHBhcmFtcyA9IHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBwZXJzaXN0ZW50OiB7XG4gICAgICAgICAgICAgIFtgc2VhcmNoLmluc2lnaHRzLnRvcF9xdWVyaWVzLiR7cXVlcnkubWV0cmljfS5lbmFibGVkYF06IHF1ZXJ5LmVuYWJsZWQsXG4gICAgICAgICAgICAgIFtgc2VhcmNoLmluc2lnaHRzLnRvcF9xdWVyaWVzLiR7cXVlcnkubWV0cmljfS50b3Bfbl9zaXplYF06IHF1ZXJ5LnRvcF9uX3NpemUsXG4gICAgICAgICAgICAgIFtgc2VhcmNoLmluc2lnaHRzLnRvcF9xdWVyaWVzLiR7cXVlcnkubWV0cmljfS53aW5kb3dfc2l6ZWBdOiBxdWVyeS53aW5kb3dfc2l6ZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgICAgaWYgKHF1ZXJ5Lmdyb3VwX2J5ICE9PSAnJykge1xuICAgICAgICAgIHBhcmFtcy5ib2R5LnBlcnNpc3RlbnRbJ3NlYXJjaC5pbnNpZ2h0cy50b3BfcXVlcmllcy5ncm91cGluZy5ncm91cF9ieSddID0gcXVlcnkuZ3JvdXBfYnk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHF1ZXJ5LmRlbGV0ZV9hZnRlcl9kYXlzICE9PSAnJykge1xuICAgICAgICAgIHBhcmFtcy5ib2R5LnBlcnNpc3RlbnRbJ3NlYXJjaC5pbnNpZ2h0cy50b3BfcXVlcmllcy5leHBvcnRlci5kZWxldGVfYWZ0ZXJfZGF5cyddID1cbiAgICAgICAgICAgIHF1ZXJ5LmRlbGV0ZV9hZnRlcl9kYXlzO1xuICAgICAgICB9XG4gICAgICAgIGlmIChxdWVyeS5leHBvcnRlclR5cGUgIT09ICcnKSB7XG4gICAgICAgICAgcGFyYW1zLmJvZHkucGVyc2lzdGVudFsnc2VhcmNoLmluc2lnaHRzLnRvcF9xdWVyaWVzLmV4cG9ydGVyLnR5cGUnXSA9XG4gICAgICAgICAgICBxdWVyeS5leHBvcnRlclR5cGUgPT09IEVYUE9SVEVSX1RZUEUubG9jYWxJbmRleFxuICAgICAgICAgICAgICA/IHF1ZXJ5LmV4cG9ydGVyVHlwZVxuICAgICAgICAgICAgICA6IEVYUE9SVEVSX1RZUEUubm9uZTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoIWRhdGFTb3VyY2VFbmFibGVkIHx8ICFyZXF1ZXN0LnF1ZXJ5Py5kYXRhU291cmNlSWQpIHtcbiAgICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LnF1ZXJ5SW5zaWdodHNfcGx1Z2luLnF1ZXJ5SW5zaWdodHNDbGllbnQuYXNTY29wZWQocmVxdWVzdClcbiAgICAgICAgICAgIC5jYWxsQXNDdXJyZW50VXNlcjtcbiAgICAgICAgICBjb25zdCByZXMgPSBhd2FpdCBjbGllbnQoJ3F1ZXJ5SW5zaWdodHMuc2V0U2V0dGluZ3MnLCBwYXJhbXMpO1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICAgICAgcmVzcG9uc2U6IHJlcyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5kYXRhU291cmNlLm9wZW5zZWFyY2gubGVnYWN5LmdldENsaWVudChcbiAgICAgICAgICAgIHJlcXVlc3QucXVlcnk/LmRhdGFTb3VyY2VJZFxuICAgICAgICAgICk7XG4gICAgICAgICAgY29uc3QgcmVzID0gYXdhaXQgY2xpZW50LmNhbGxBUEkoJ3F1ZXJ5SW5zaWdodHMuc2V0U2V0dGluZ3MnLCBwYXJhbXMpO1xuICAgICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgICAgc3RhdHVzQ29kZTogMjAwLFxuICAgICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICAgICAgcmVzcG9uc2U6IHJlcyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ1VuYWJsZSB0byBzZXQgc2V0dGluZ3M6ICcsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgICByZXNwb25zZTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogJy9hcGkvbGl2ZV9xdWVyaWVzJyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHF1ZXJ5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBkYXRhU291cmNlSWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoY29udGV4dCwgcmVxdWVzdCwgcmVzcG9uc2UpID0+IHtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGNsaWVudCA9XG4gICAgICAgICAgIWRhdGFTb3VyY2VFbmFibGVkIHx8ICFyZXF1ZXN0LnF1ZXJ5Py5kYXRhU291cmNlSWRcbiAgICAgICAgICAgID8gY29udGV4dC5xdWVyeUluc2lnaHRzX3BsdWdpbi5xdWVyeUluc2lnaHRzQ2xpZW50LmFzU2NvcGVkKHJlcXVlc3QpLmNhbGxBc0N1cnJlbnRVc2VyXG4gICAgICAgICAgICA6IGNvbnRleHQuZGF0YVNvdXJjZS5vcGVuc2VhcmNoLmxlZ2FjeS5nZXRDbGllbnQocmVxdWVzdC5xdWVyeS5kYXRhU291cmNlSWQpO1xuXG4gICAgICAgIGNvbnN0IHJlcyA9IGF3YWl0IGNsaWVudCgncXVlcnlJbnNpZ2h0cy5nZXRMaXZlUXVlcmllcycpO1xuICAgICAgICBpZiAoIXJlcyB8fCByZXMub2sgPT09IGZhbHNlKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKHJlcz8uZXJyb3IgfHwgJ1F1ZXJ5IEluc2lnaHRzIHNlcnZpY2UgcmV0dXJuZWQgYW4gZXJyb3InKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgICByZXNwb25zZTogcmVzLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignVW5hYmxlIHRvIGdldCBsaXZlIHF1ZXJpZXM6ICcsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbUVycm9yKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlID8/IDUwMCxcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiBlcnJvci5tZXNzYWdlIHx8ICdJbnRlcm5hbCBzZXJ2ZXIgZXJyb3InLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiAnL2FwaS90YXNrcy97dGFza0lkfS9jYW5jZWwnLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICB0YXNrSWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBlc0NsaWVudCA9IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmNsaWVudC5hc0N1cnJlbnRVc2VyO1xuICAgICAgICBjb25zdCB7IHRhc2tJZCB9ID0gcmVxdWVzdC5wYXJhbXM7XG5cbiAgICAgICAgY29uc3QgcmVzID0gYXdhaXQgZXNDbGllbnQudHJhbnNwb3J0LnJlcXVlc3Qoe1xuICAgICAgICAgIG1ldGhvZDogJ1BPU1QnLFxuICAgICAgICAgIHBhdGg6IGAvX3Rhc2tzLyR7dGFza0lkfS9fY2FuY2VsYCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgaWYgKCFyZXMpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2ZhaWxlZCcpO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHsgYm9keTogeyBvazogdHJ1ZSwgcmVzIH0gfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICBjb25zb2xlLmVycm9yKGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbUVycm9yKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlID8/IDUwMCxcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiBlcnJvci5tZXNzYWdlIHx8ICdJbnRlcm5hbCBzZXJ2ZXIgZXJyb3InLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBQUEsYUFBQSxHQUFBQyxPQUFBO0FBRUEsSUFBQUMsVUFBQSxHQUFBRCxPQUFBO0FBUEE7QUFDQTtBQUNBO0FBQ0E7O0FBTU8sU0FBU0UsWUFBWUEsQ0FBQ0MsTUFBZSxFQUFFQyxpQkFBMEIsRUFBRTtFQUN4RUQsTUFBTSxDQUFDRSxHQUFHLENBQ1I7SUFDRUMsSUFBSSxFQUFFLGtCQUFrQjtJQUN4QkMsUUFBUSxFQUFFO01BQ1JDLEtBQUssRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ25CQyxZQUFZLEVBQUVGLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDLENBQUM7TUFDNUMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU9DLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQUs7SUFDcEMsSUFBSTtNQUFBLElBQUFDLGNBQUE7TUFDRixJQUFJLENBQUNiLGlCQUFpQixJQUFJLEdBQUFhLGNBQUEsR0FBQ0YsT0FBTyxDQUFDUCxLQUFLLGNBQUFTLGNBQUEsZUFBYkEsY0FBQSxDQUFlTixZQUFZLEdBQUU7UUFDdEQsTUFBTU8sTUFBTSxHQUFHSixPQUFPLENBQUNLLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUFDTixPQUFPLENBQUMsQ0FDOUVPLGlCQUFpQjtRQUNwQixNQUFNQyxHQUFHLEdBQUcsTUFBTUwsTUFBTSxDQUFDLDhCQUE4QixDQUFDO1FBQ3hELE9BQU9GLFFBQVEsQ0FBQ1EsTUFBTSxDQUFDO1VBQ3JCQyxVQUFVLEVBQUUsR0FBRztVQUNmQyxJQUFJLEVBQUU7WUFDSkMsRUFBRSxFQUFFLElBQUk7WUFDUlgsUUFBUSxFQUFFTztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxNQUFNO1FBQUEsSUFBQUssZUFBQTtRQUNMLE1BQU1WLE1BQU0sR0FBR0osT0FBTyxDQUFDZSxVQUFVLENBQUNDLFVBQVUsQ0FBQ0MsTUFBTSxDQUFDQyxTQUFTLEVBQUFKLGVBQUEsR0FDM0RiLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBb0IsZUFBQSx1QkFBYkEsZUFBQSxDQUFlakIsWUFDakIsQ0FBQztRQUNELE1BQU1ZLEdBQUcsR0FBRyxNQUFNTCxNQUFNLENBQUNlLE9BQU8sQ0FBQyw4QkFBOEIsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNwRSxPQUFPakIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUMsQ0FBQyxPQUFPVyxLQUFLLEVBQUU7TUFDZEMsT0FBTyxDQUFDRCxLQUFLLENBQUMsNkJBQTZCLEVBQUVBLEtBQUssQ0FBQztNQUNuRCxPQUFPbEIsUUFBUSxDQUFDVyxFQUFFLENBQUM7UUFDakJELElBQUksRUFBRTtVQUNKQyxFQUFFLEVBQUUsS0FBSztVQUNUWCxRQUFRLEVBQUVrQixLQUFLLENBQUNFO1FBQ2xCO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7RUFFRGpDLE1BQU0sQ0FBQ0UsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSwwQkFBMEI7SUFDaENDLFFBQVEsRUFBRTtNQUNSQyxLQUFLLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNuQjJCLElBQUksRUFBRTVCLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDO1VBQUV5QixZQUFZLEVBQUU7UUFBRyxDQUFDLENBQUMsQ0FBQztRQUN2REMsRUFBRSxFQUFFOUIsb0JBQU0sQ0FBQ0csS0FBSyxDQUFDSCxvQkFBTSxDQUFDSSxNQUFNLENBQUM7VUFBRXlCLFlBQVksRUFBRTtRQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3JERSxFQUFFLEVBQUUvQixvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDckQzQixZQUFZLEVBQUVGLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDLENBQUMsQ0FBQztRQUMzQzRCLE9BQU8sRUFBRWhDLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ2lDLE9BQU8sQ0FBQztVQUFFSixZQUFZLEVBQUU7UUFBTSxDQUFDLENBQUM7TUFDL0QsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU94QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFBQSxJQUFBMkIsZUFBQTtNQUNGLE1BQU07UUFBRU4sSUFBSTtRQUFFRSxFQUFFO1FBQUVDLEVBQUU7UUFBRUM7TUFBUSxDQUFDLEdBQUcxQixPQUFPLENBQUNQLEtBQUs7TUFDL0MsTUFBTW9DLE1BQU0sR0FBRztRQUFFUCxJQUFJO1FBQUVFLEVBQUU7UUFBRUMsRUFBRTtRQUFFQztNQUFRLENBQUM7TUFDeEMsSUFBSSxDQUFDckMsaUJBQWlCLElBQUksR0FBQXVDLGVBQUEsR0FBQzVCLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBbUMsZUFBQSxlQUFiQSxlQUFBLENBQWVoQyxZQUFZLEdBQUU7UUFDdEQsTUFBTU8sTUFBTSxHQUFHSixPQUFPLENBQUNLLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUFDTixPQUFPLENBQUMsQ0FDOUVPLGlCQUFpQjtRQUNwQixNQUFNQyxHQUFHLEdBQ1BpQixFQUFFLElBQUksSUFBSSxHQUNOLE1BQU10QixNQUFNLENBQUMsMENBQTBDLEVBQUUwQixNQUFNLENBQUMsR0FDaEUsTUFBTTFCLE1BQU0sQ0FBQyxxQ0FBcUMsRUFBRTBCLE1BQU0sQ0FBQztRQUNqRSxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLE1BQU07UUFBQSxJQUFBc0IsZUFBQTtRQUNMLE1BQU0zQixNQUFNLEdBQUdKLE9BQU8sQ0FBQ2UsVUFBVSxDQUFDQyxVQUFVLENBQUNDLE1BQU0sQ0FBQ0MsU0FBUyxFQUFBYSxlQUFBLEdBQzNEOUIsT0FBTyxDQUFDUCxLQUFLLGNBQUFxQyxlQUFBLHVCQUFiQSxlQUFBLENBQWVsQyxZQUNqQixDQUFDO1FBQ0QsTUFBTVksR0FBRyxHQUNQaUIsRUFBRSxJQUFJLElBQUksR0FDTixNQUFNdEIsTUFBTSxDQUFDZSxPQUFPLENBQUMsMENBQTBDLEVBQUVXLE1BQU0sQ0FBQyxHQUN4RSxNQUFNMUIsTUFBTSxDQUFDZSxPQUFPLENBQUMscUNBQXFDLEVBQUVXLE1BQU0sQ0FBQztRQUN6RSxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUMsQ0FBQyxPQUFPVyxLQUFLLEVBQUU7TUFDZEMsT0FBTyxDQUFDRCxLQUFLLENBQUMsdUNBQXVDLEVBQUVBLEtBQUssQ0FBQztNQUM3RCxPQUFPbEIsUUFBUSxDQUFDVyxFQUFFLENBQUM7UUFDakJELElBQUksRUFBRTtVQUNKQyxFQUFFLEVBQUUsS0FBSztVQUNUWCxRQUFRLEVBQUVrQixLQUFLLENBQUNFO1FBQ2xCO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7RUFFRGpDLE1BQU0sQ0FBQ0UsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSxzQkFBc0I7SUFDNUJDLFFBQVEsRUFBRTtNQUNSQyxLQUFLLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNuQjJCLElBQUksRUFBRTVCLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDO1VBQUV5QixZQUFZLEVBQUU7UUFBRyxDQUFDLENBQUMsQ0FBQztRQUN2REMsRUFBRSxFQUFFOUIsb0JBQU0sQ0FBQ0csS0FBSyxDQUFDSCxvQkFBTSxDQUFDSSxNQUFNLENBQUM7VUFBRXlCLFlBQVksRUFBRTtRQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3JERSxFQUFFLEVBQUUvQixvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDckQzQixZQUFZLEVBQUVGLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDLENBQUMsQ0FBQztRQUMzQzRCLE9BQU8sRUFBRWhDLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ2lDLE9BQU8sQ0FBQztVQUFFSixZQUFZLEVBQUU7UUFBTSxDQUFDLENBQUM7TUFDL0QsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU94QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFBQSxJQUFBOEIsZUFBQTtNQUNGLE1BQU07UUFBRVQsSUFBSTtRQUFFRSxFQUFFO1FBQUVDLEVBQUU7UUFBRUM7TUFBUSxDQUFDLEdBQUcxQixPQUFPLENBQUNQLEtBQUs7TUFDL0MsTUFBTW9DLE1BQU0sR0FBRztRQUFFUCxJQUFJO1FBQUVFLEVBQUU7UUFBRUMsRUFBRTtRQUFFQztNQUFRLENBQUM7TUFFeEMsSUFBSSxDQUFDckMsaUJBQWlCLElBQUksR0FBQTBDLGVBQUEsR0FBQy9CLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBc0MsZUFBQSxlQUFiQSxlQUFBLENBQWVuQyxZQUFZLEdBQUU7UUFDdEQsTUFBTU8sTUFBTSxHQUFHSixPQUFPLENBQUNLLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUFDTixPQUFPLENBQUMsQ0FDOUVPLGlCQUFpQjtRQUNwQixNQUFNQyxHQUFHLEdBQ1BpQixFQUFFLElBQUksSUFBSSxHQUNOLE1BQU10QixNQUFNLENBQUMsc0NBQXNDLEVBQUUwQixNQUFNLENBQUMsR0FDNUQsTUFBTTFCLE1BQU0sQ0FBQyxpQ0FBaUMsRUFBRTBCLE1BQU0sQ0FBQztRQUM3RCxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLE1BQU07UUFBQSxJQUFBd0IsZUFBQTtRQUNMLE1BQU03QixNQUFNLEdBQUdKLE9BQU8sQ0FBQ2UsVUFBVSxDQUFDQyxVQUFVLENBQUNDLE1BQU0sQ0FBQ0MsU0FBUyxFQUFBZSxlQUFBLEdBQzNEaEMsT0FBTyxDQUFDUCxLQUFLLGNBQUF1QyxlQUFBLHVCQUFiQSxlQUFBLENBQWVwQyxZQUNqQixDQUFDO1FBQ0QsTUFBTVksR0FBRyxHQUNQaUIsRUFBRSxJQUFJLElBQUksR0FDTixNQUFNdEIsTUFBTSxDQUFDZSxPQUFPLENBQUMsc0NBQXNDLEVBQUVXLE1BQU0sQ0FBQyxHQUNwRSxNQUFNMUIsTUFBTSxDQUFDZSxPQUFPLENBQUMsaUNBQWlDLEVBQUVXLE1BQU0sQ0FBQztRQUNyRSxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUMsQ0FBQyxPQUFPVyxLQUFLLEVBQUU7TUFDZEMsT0FBTyxDQUFDRCxLQUFLLENBQUMsbUNBQW1DLEVBQUVBLEtBQUssQ0FBQztNQUN6RCxPQUFPbEIsUUFBUSxDQUFDVyxFQUFFLENBQUM7UUFDakJELElBQUksRUFBRTtVQUNKQyxFQUFFLEVBQUUsS0FBSztVQUNUWCxRQUFRLEVBQUVrQixLQUFLLENBQUNFO1FBQ2xCO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7RUFFRGpDLE1BQU0sQ0FBQ0UsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSx5QkFBeUI7SUFDL0JDLFFBQVEsRUFBRTtNQUNSQyxLQUFLLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNuQjJCLElBQUksRUFBRTVCLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDO1VBQUV5QixZQUFZLEVBQUU7UUFBRyxDQUFDLENBQUMsQ0FBQztRQUN2REMsRUFBRSxFQUFFOUIsb0JBQU0sQ0FBQ0csS0FBSyxDQUFDSCxvQkFBTSxDQUFDSSxNQUFNLENBQUM7VUFBRXlCLFlBQVksRUFBRTtRQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3JERSxFQUFFLEVBQUUvQixvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDckQzQixZQUFZLEVBQUVGLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDLENBQUMsQ0FBQztRQUMzQzRCLE9BQU8sRUFBRWhDLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ2lDLE9BQU8sQ0FBQztVQUFFSixZQUFZLEVBQUU7UUFBTSxDQUFDLENBQUM7TUFDL0QsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU94QixPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFBQSxJQUFBZ0MsZUFBQTtNQUNGLE1BQU07UUFBRVgsSUFBSTtRQUFFRSxFQUFFO1FBQUVDLEVBQUU7UUFBRUM7TUFBUSxDQUFDLEdBQUcxQixPQUFPLENBQUNQLEtBQUs7TUFDL0MsTUFBTW9DLE1BQU0sR0FBRztRQUFFUCxJQUFJO1FBQUVFLEVBQUU7UUFBRUMsRUFBRTtRQUFFQztNQUFRLENBQUM7TUFDeEMsSUFBSSxDQUFDckMsaUJBQWlCLElBQUksR0FBQTRDLGVBQUEsR0FBQ2pDLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBd0MsZUFBQSxlQUFiQSxlQUFBLENBQWVyQyxZQUFZLEdBQUU7UUFDdEQsTUFBTU8sTUFBTSxHQUFHSixPQUFPLENBQUNLLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUFDTixPQUFPLENBQUMsQ0FDOUVPLGlCQUFpQjtRQUNwQixNQUFNQyxHQUFHLEdBQ1BpQixFQUFFLElBQUksSUFBSSxHQUNOLE1BQU10QixNQUFNLENBQUMseUNBQXlDLEVBQUUwQixNQUFNLENBQUMsR0FDL0QsTUFBTTFCLE1BQU0sQ0FBQyxvQ0FBb0MsRUFBRTBCLE1BQU0sQ0FBQztRQUNoRSxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLE1BQU07UUFBQSxJQUFBMEIsZUFBQTtRQUNMLE1BQU0vQixNQUFNLEdBQUdKLE9BQU8sQ0FBQ2UsVUFBVSxDQUFDQyxVQUFVLENBQUNDLE1BQU0sQ0FBQ0MsU0FBUyxFQUFBaUIsZUFBQSxHQUMzRGxDLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBeUMsZUFBQSx1QkFBYkEsZUFBQSxDQUFldEMsWUFDakIsQ0FBQztRQUNELE1BQU1ZLEdBQUcsR0FDUGlCLEVBQUUsSUFBSSxJQUFJLEdBQ04sTUFBTXRCLE1BQU0sQ0FBQ2UsT0FBTyxDQUFDLHlDQUF5QyxFQUFFVyxNQUFNLENBQUMsR0FDdkUsTUFBTTFCLE1BQU0sQ0FBQ2UsT0FBTyxDQUFDLG9DQUFvQyxFQUFFVyxNQUFNLENBQUM7UUFDeEUsT0FBTzVCLFFBQVEsQ0FBQ1EsTUFBTSxDQUFDO1VBQ3JCQyxVQUFVLEVBQUUsR0FBRztVQUNmQyxJQUFJLEVBQUU7WUFDSkMsRUFBRSxFQUFFLElBQUk7WUFDUlgsUUFBUSxFQUFFTztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDLENBQUMsT0FBT1csS0FBSyxFQUFFO01BQ2RDLE9BQU8sQ0FBQ0QsS0FBSyxDQUFDLHNDQUFzQyxFQUFFQSxLQUFLLENBQUM7TUFDNUQsT0FBT2xCLFFBQVEsQ0FBQ1csRUFBRSxDQUFDO1FBQ2pCRCxJQUFJLEVBQUU7VUFDSkMsRUFBRSxFQUFFLEtBQUs7VUFDVFgsUUFBUSxFQUFFa0IsS0FBSyxDQUFDRTtRQUNsQjtNQUNGLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDO0VBRURqQyxNQUFNLENBQUNFLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUUsZUFBZTtJQUNyQkMsUUFBUSxFQUFFO01BQ1JDLEtBQUssRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ25CQyxZQUFZLEVBQUVGLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDLENBQUM7TUFDNUMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU9DLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQUs7SUFDcEMsSUFBSTtNQUFBLElBQUFrQyxlQUFBO01BQ0YsSUFBSSxDQUFDOUMsaUJBQWlCLElBQUksR0FBQThDLGVBQUEsR0FBQ25DLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBMEMsZUFBQSxlQUFiQSxlQUFBLENBQWV2QyxZQUFZLEdBQUU7UUFDdEQsTUFBTU8sTUFBTSxHQUFHSixPQUFPLENBQUNLLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUFDTixPQUFPLENBQUMsQ0FDOUVPLGlCQUFpQjtRQUNwQixNQUFNQyxHQUFHLEdBQUcsTUFBTUwsTUFBTSxDQUFDLDJCQUEyQixDQUFDO1FBQ3JELE9BQU9GLFFBQVEsQ0FBQ1EsTUFBTSxDQUFDO1VBQ3JCQyxVQUFVLEVBQUUsR0FBRztVQUNmQyxJQUFJLEVBQUU7WUFDSkMsRUFBRSxFQUFFLElBQUk7WUFDUlgsUUFBUSxFQUFFTztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxNQUFNO1FBQUEsSUFBQTRCLGdCQUFBO1FBQ0wsTUFBTWpDLE1BQU0sR0FBR0osT0FBTyxDQUFDZSxVQUFVLENBQUNDLFVBQVUsQ0FBQ0MsTUFBTSxDQUFDQyxTQUFTLEVBQUFtQixnQkFBQSxHQUMzRHBDLE9BQU8sQ0FBQ1AsS0FBSyxjQUFBMkMsZ0JBQUEsdUJBQWJBLGdCQUFBLENBQWV4QyxZQUNqQixDQUFDO1FBQ0QsTUFBTVksR0FBRyxHQUFHLE1BQU1MLE1BQU0sQ0FBQ2UsT0FBTyxDQUFDLDJCQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2pFLE9BQU9qQixRQUFRLENBQUNRLE1BQU0sQ0FBQztVQUNyQkMsVUFBVSxFQUFFLEdBQUc7VUFDZkMsSUFBSSxFQUFFO1lBQ0pDLEVBQUUsRUFBRSxJQUFJO1lBQ1JYLFFBQVEsRUFBRU87VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQyxDQUFDLE9BQU9XLEtBQUssRUFBRTtNQUNkQyxPQUFPLENBQUNELEtBQUssQ0FBQyw2QkFBNkIsRUFBRUEsS0FBSyxDQUFDO01BQ25ELE9BQU9sQixRQUFRLENBQUNXLEVBQUUsQ0FBQztRQUNqQkQsSUFBSSxFQUFFO1VBQ0pDLEVBQUUsRUFBRSxLQUFLO1VBQ1RYLFFBQVEsRUFBRWtCLEtBQUssQ0FBQ0U7UUFDbEI7TUFDRixDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQztFQUVEakMsTUFBTSxDQUFDaUQsR0FBRyxDQUNSO0lBQ0U5QyxJQUFJLEVBQUUsc0JBQXNCO0lBQzVCQyxRQUFRLEVBQUU7TUFDUkMsS0FBSyxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDbkIyQyxNQUFNLEVBQUU1QyxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDekRnQixPQUFPLEVBQUU3QyxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNpQyxPQUFPLENBQUM7VUFBRUosWUFBWSxFQUFFO1FBQU0sQ0FBQyxDQUFDLENBQUM7UUFDOURpQixVQUFVLEVBQUU5QyxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDN0RrQixXQUFXLEVBQUUvQyxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDOURtQixZQUFZLEVBQUVoRCxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDL0RvQixRQUFRLEVBQUVqRCxvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQztVQUFFeUIsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDLENBQUM7UUFDM0QzQixZQUFZLEVBQUVGLG9CQUFNLENBQUNHLEtBQUssQ0FBQ0gsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDLENBQUMsQ0FBQztRQUMzQzhDLGlCQUFpQixFQUFFbEQsb0JBQU0sQ0FBQ0csS0FBSyxDQUFDSCxvQkFBTSxDQUFDSSxNQUFNLENBQUM7VUFBRXlCLFlBQVksRUFBRTtRQUFHLENBQUMsQ0FBQztNQUNyRSxDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FBT3hCLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQUs7SUFDcEMsSUFBSTtNQUFBLElBQUE0QyxnQkFBQTtNQUNGLE1BQU1wRCxLQUFLLEdBQUdPLE9BQU8sQ0FBQ1AsS0FBSztNQUMzQixNQUFNb0MsTUFBTSxHQUFHO1FBQ2JsQixJQUFJLEVBQUU7VUFDSm1DLFVBQVUsRUFBRTtZQUNWLENBQUUsK0JBQThCckQsS0FBSyxDQUFDNkMsTUFBTyxVQUFTLEdBQUc3QyxLQUFLLENBQUM4QyxPQUFPO1lBQ3RFLENBQUUsK0JBQThCOUMsS0FBSyxDQUFDNkMsTUFBTyxhQUFZLEdBQUc3QyxLQUFLLENBQUMrQyxVQUFVO1lBQzVFLENBQUUsK0JBQThCL0MsS0FBSyxDQUFDNkMsTUFBTyxjQUFhLEdBQUc3QyxLQUFLLENBQUNnRDtVQUNyRTtRQUNGO01BQ0YsQ0FBQztNQUNELElBQUloRCxLQUFLLENBQUNrRCxRQUFRLEtBQUssRUFBRSxFQUFFO1FBQ3pCZCxNQUFNLENBQUNsQixJQUFJLENBQUNtQyxVQUFVLENBQUMsK0NBQStDLENBQUMsR0FBR3JELEtBQUssQ0FBQ2tELFFBQVE7TUFDMUY7TUFDQSxJQUFJbEQsS0FBSyxDQUFDbUQsaUJBQWlCLEtBQUssRUFBRSxFQUFFO1FBQ2xDZixNQUFNLENBQUNsQixJQUFJLENBQUNtQyxVQUFVLENBQUMsd0RBQXdELENBQUMsR0FDOUVyRCxLQUFLLENBQUNtRCxpQkFBaUI7TUFDM0I7TUFDQSxJQUFJbkQsS0FBSyxDQUFDaUQsWUFBWSxLQUFLLEVBQUUsRUFBRTtRQUM3QmIsTUFBTSxDQUFDbEIsSUFBSSxDQUFDbUMsVUFBVSxDQUFDLDJDQUEyQyxDQUFDLEdBQ2pFckQsS0FBSyxDQUFDaUQsWUFBWSxLQUFLSyx3QkFBYSxDQUFDQyxVQUFVLEdBQzNDdkQsS0FBSyxDQUFDaUQsWUFBWSxHQUNsQkssd0JBQWEsQ0FBQ0UsSUFBSTtNQUMxQjtNQUNBLElBQUksQ0FBQzVELGlCQUFpQixJQUFJLEdBQUF3RCxnQkFBQSxHQUFDN0MsT0FBTyxDQUFDUCxLQUFLLGNBQUFvRCxnQkFBQSxlQUFiQSxnQkFBQSxDQUFlakQsWUFBWSxHQUFFO1FBQ3RELE1BQU1PLE1BQU0sR0FBR0osT0FBTyxDQUFDSyxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FBQ04sT0FBTyxDQUFDLENBQzlFTyxpQkFBaUI7UUFDcEIsTUFBTUMsR0FBRyxHQUFHLE1BQU1MLE1BQU0sQ0FBQywyQkFBMkIsRUFBRTBCLE1BQU0sQ0FBQztRQUM3RCxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLE1BQU07UUFBQSxJQUFBMEMsZ0JBQUE7UUFDTCxNQUFNL0MsTUFBTSxHQUFHSixPQUFPLENBQUNlLFVBQVUsQ0FBQ0MsVUFBVSxDQUFDQyxNQUFNLENBQUNDLFNBQVMsRUFBQWlDLGdCQUFBLEdBQzNEbEQsT0FBTyxDQUFDUCxLQUFLLGNBQUF5RCxnQkFBQSx1QkFBYkEsZ0JBQUEsQ0FBZXRELFlBQ2pCLENBQUM7UUFDRCxNQUFNWSxHQUFHLEdBQUcsTUFBTUwsTUFBTSxDQUFDZSxPQUFPLENBQUMsMkJBQTJCLEVBQUVXLE1BQU0sQ0FBQztRQUNyRSxPQUFPNUIsUUFBUSxDQUFDUSxNQUFNLENBQUM7VUFDckJDLFVBQVUsRUFBRSxHQUFHO1VBQ2ZDLElBQUksRUFBRTtZQUNKQyxFQUFFLEVBQUUsSUFBSTtZQUNSWCxRQUFRLEVBQUVPO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUMsQ0FBQyxPQUFPVyxLQUFLLEVBQUU7TUFDZEMsT0FBTyxDQUFDRCxLQUFLLENBQUMsMEJBQTBCLEVBQUVBLEtBQUssQ0FBQztNQUNoRCxPQUFPbEIsUUFBUSxDQUFDVyxFQUFFLENBQUM7UUFDakJELElBQUksRUFBRTtVQUNKQyxFQUFFLEVBQUUsS0FBSztVQUNUWCxRQUFRLEVBQUVrQixLQUFLLENBQUNFO1FBQ2xCO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7RUFDRGpDLE1BQU0sQ0FBQ0UsR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRSxtQkFBbUI7SUFDekJDLFFBQVEsRUFBRTtNQUNSQyxLQUFLLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNuQkMsWUFBWSxFQUFFRixvQkFBTSxDQUFDRyxLQUFLLENBQUNILG9CQUFNLENBQUNJLE1BQU0sQ0FBQyxDQUFDO01BQzVDLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPQyxPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFBQSxJQUFBa0QsZ0JBQUE7TUFDRixNQUFNaEQsTUFBTSxHQUNWLENBQUNkLGlCQUFpQixJQUFJLEdBQUE4RCxnQkFBQSxHQUFDbkQsT0FBTyxDQUFDUCxLQUFLLGNBQUEwRCxnQkFBQSxlQUFiQSxnQkFBQSxDQUFldkQsWUFBWSxJQUM5Q0csT0FBTyxDQUFDSyxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FBQ04sT0FBTyxDQUFDLENBQUNPLGlCQUFpQixHQUNwRlIsT0FBTyxDQUFDZSxVQUFVLENBQUNDLFVBQVUsQ0FBQ0MsTUFBTSxDQUFDQyxTQUFTLENBQUNqQixPQUFPLENBQUNQLEtBQUssQ0FBQ0csWUFBWSxDQUFDO01BRWhGLE1BQU1ZLEdBQUcsR0FBRyxNQUFNTCxNQUFNLENBQUMsOEJBQThCLENBQUM7TUFDeEQsSUFBSSxDQUFDSyxHQUFHLElBQUlBLEdBQUcsQ0FBQ0ksRUFBRSxLQUFLLEtBQUssRUFBRTtRQUM1QixNQUFNLElBQUl3QyxLQUFLLENBQUMsQ0FBQTVDLEdBQUcsYUFBSEEsR0FBRyx1QkFBSEEsR0FBRyxDQUFFVyxLQUFLLEtBQUksMENBQTBDLENBQUM7TUFDM0U7TUFFQSxPQUFPbEIsUUFBUSxDQUFDVyxFQUFFLENBQUM7UUFDakJELElBQUksRUFBRTtVQUNKQyxFQUFFLEVBQUUsSUFBSTtVQUNSWCxRQUFRLEVBQUVPO1FBQ1o7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT1csS0FBSyxFQUFFO01BQUEsSUFBQWtDLGlCQUFBO01BQ2RqQyxPQUFPLENBQUNELEtBQUssQ0FBQyw4QkFBOEIsRUFBRUEsS0FBSyxDQUFDO01BQ3BELE9BQU9sQixRQUFRLENBQUNxRCxXQUFXLENBQUM7UUFDMUI1QyxVQUFVLEdBQUEyQyxpQkFBQSxHQUFFbEMsS0FBSyxDQUFDVCxVQUFVLGNBQUEyQyxpQkFBQSxjQUFBQSxpQkFBQSxHQUFJLEdBQUc7UUFDbkMxQyxJQUFJLEVBQUU7VUFDSlUsT0FBTyxFQUFFRixLQUFLLENBQUNFLE9BQU8sSUFBSTtRQUM1QjtNQUNGLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDO0VBRURqQyxNQUFNLENBQUNtRSxJQUFJLENBQ1Q7SUFDRWhFLElBQUksRUFBRSw0QkFBNEI7SUFDbENDLFFBQVEsRUFBRTtNQUNScUMsTUFBTSxFQUFFbkMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCNkQsTUFBTSxFQUFFOUQsb0JBQU0sQ0FBQ0ksTUFBTSxDQUFDO01BQ3hCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPQyxPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLElBQUk7TUFDRixNQUFNd0QsUUFBUSxHQUFHMUQsT0FBTyxDQUFDMkQsSUFBSSxDQUFDM0MsVUFBVSxDQUFDWixNQUFNLENBQUN3RCxhQUFhO01BQzdELE1BQU07UUFBRUg7TUFBTyxDQUFDLEdBQUd4RCxPQUFPLENBQUM2QixNQUFNO01BRWpDLE1BQU1yQixHQUFHLEdBQUcsTUFBTWlELFFBQVEsQ0FBQ0csU0FBUyxDQUFDNUQsT0FBTyxDQUFDO1FBQzNDNkQsTUFBTSxFQUFFLE1BQU07UUFDZHRFLElBQUksRUFBRyxXQUFVaUUsTUFBTztNQUMxQixDQUFDLENBQUM7TUFFRixJQUFJLENBQUNoRCxHQUFHLEVBQUU7UUFDUixNQUFNLElBQUk0QyxLQUFLLENBQUMsUUFBUSxDQUFDO01BQzNCO01BRUEsT0FBT25ELFFBQVEsQ0FBQ1csRUFBRSxDQUFDO1FBQUVELElBQUksRUFBRTtVQUFFQyxFQUFFLEVBQUUsSUFBSTtVQUFFSjtRQUFJO01BQUUsQ0FBQyxDQUFDO0lBQ2pELENBQUMsQ0FBQyxPQUFPVyxLQUFLLEVBQUU7TUFBQSxJQUFBMkMsa0JBQUE7TUFDZDFDLE9BQU8sQ0FBQ0QsS0FBSyxDQUFDQSxLQUFLLENBQUM7TUFDcEIsT0FBT2xCLFFBQVEsQ0FBQ3FELFdBQVcsQ0FBQztRQUMxQjVDLFVBQVUsR0FBQW9ELGtCQUFBLEdBQUUzQyxLQUFLLENBQUNULFVBQVUsY0FBQW9ELGtCQUFBLGNBQUFBLGtCQUFBLEdBQUksR0FBRztRQUNuQ25ELElBQUksRUFBRTtVQUNKVSxPQUFPLEVBQUVGLEtBQUssQ0FBQ0UsT0FBTyxJQUFJO1FBQzVCO01BQ0YsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7QUFDSCJ9