dnl This is m4 source.
dnl Process using m4 to produce 'C' language file.
dnl
dnl This file is supposed to be the same as PnetCDF's test_read.m4
dnl
dnl If you see this line, you can ignore the next one.
/* Do not edit this file. It is produced from the corresponding .m4 source */
dnl
/*
 *  Copyright (C) 2003, Northwestern University and Argonne National Laboratory
 *  See COPYRIGHT notice in top-level directory.
 */
/* $Id$ */

dnl
dnl The command-line m4 macro "PNETCDF" is to differentiate PnetCDF and netCDF
dnl in terms of function prefix names (ncmpi_ vs. nc_), integer data types
dnl (MPI_Offset vs. size_t), and function name substrings for external data
dnl types.
dnl

#include <sys/types.h> /* open() */
#include <sys/stat.h>  /* open() */
#include <fcntl.h>     /* open() */
#include <unistd.h>    /* unlink(), write() */
#include <errno.h>     /* errno, strerror() */

#include "tests.h"

ifdef(`PNETCDF',`dnl
#define NC_ERR_CODE_NAME ncmpi_strerrno',`dnl
#define NC_ERR_CODE_NAME nc_err_code_name')

define(`EXPECT_ERR',`error("expecting $1 but got %s",NC_ERR_CODE_NAME($2));')dnl

define(`IntType', `ifdef(`PNETCDF',`MPI_Offset',`size_t')')dnl
define(`PTRDType',`ifdef(`PNETCDF',`MPI_Offset',`ptrdiff_t')')dnl
define(`TestFunc',`ifdef(`PNETCDF',`test_ncmpi_$1',`test_nc_$1')')dnl
define(`APIFunc',` ifdef(`PNETCDF',`ncmpi_$1',`nc_$1')')dnl

define(`FileOpen', `ifdef(`PNETCDF',`ncmpi_open(comm, $1, $2, info, $3)', `file_open($1, $2, $3)')')dnl
define(`FileCreate',`ifdef(`PNETCDF',`ncmpi_create(comm, $1, $2, info, $3)', `file_create($1, $2, $3)')')dnl
define(`FileDelete',`ifdef(`PNETCDF',`ncmpi_delete($1,$2)',`nc_delete($1)')')dnl

define(`VarArgs',   `ifdef(`PNETCDF',`int numVars',`void')')dnl
define(`AttArgs',   `ifdef(`PNETCDF',`int numGatts',`void')')dnl
define(`AttVarArgs',`ifdef(`PNETCDF',`int numGatts, int numVars',`void')')dnl

define(`GetVar1', `ifdef(`PNETCDF',`ncmpi_get_var1_all($1,$2,$3,$4,$5,$6)',          `nc_get_var1($1,$2,$3,$4)')')dnl
define(`GetVar',  `ifdef(`PNETCDF',`ncmpi_get_var_all( $1,$2,$3,$4,$5)',             `nc_get_var( $1,$2,$3)')')dnl
define(`GetVara', `ifdef(`PNETCDF',`ncmpi_get_vara_all($1,$2,$3,$4,$5,$6,$7)',       `nc_get_vara($1,$2,$3,$4,$5)')')dnl
define(`GetVars', `ifdef(`PNETCDF',`ncmpi_get_vars_all($1,$2,$3,$4,$5,$6,$7,$8)',    `nc_get_vars($1,$2,$3,$4,$5,$6)')')dnl
define(`GetVarm', `ifdef(`PNETCDF',`ncmpi_get_varm_all($1,$2,$3,$4,$5,$6,$7,$8,$9)', `nc_get_varm($1,$2,$3,$4,$5,$6,$7)')')dnl

/*
 * Test APIFunc(strerror).
 *    Try on a bad error status.
 *    Test for each defined error status.
 */
int
TestFunc(strerror)(void)
{
    int i;
    const char *message, *expected_msg;
    int nok=0;

    static const struct {
        int status;
        const char *msg;
    } ncerrs[] = {
        {NC_NOERR, "No error"},
        {NC_EBADID, "NetCDF: Not a valid ID"},
        {NC_ENFILE, "NetCDF: Too many files open"},
        {NC_EEXIST, "NetCDF: File exists && NC_NOCLOBBER"},
        {NC_EINVAL, "NetCDF: Invalid argument"},
        {NC_EPERM, "NetCDF: Write to read only"},
        {NC_ENOTINDEFINE, "NetCDF: Operation not allowed in data mode"},
        {NC_EINDEFINE, "NetCDF: Operation not allowed in define mode"},
        {NC_EINVALCOORDS, "NetCDF: Index exceeds dimension bound"},
        {NC_EMAXDIMS, "NetCDF: NC_MAX_DIMS or NC_MAX_VAR_DIMS exceeded"},
        {NC_ENAMEINUSE, "NetCDF: String match to name in use"},
        {NC_ENOTATT, "NetCDF: Attribute not found"},
        {NC_EMAXATTS, "NetCDF: NC_MAX_ATTRS exceeded"},
        {NC_EBADTYPE, "NetCDF: Not a valid data type or _FillValue type mismatch"},
        {NC_EBADDIM, "NetCDF: Invalid dimension ID or name"},
        {NC_EUNLIMPOS, "NetCDF: NC_UNLIMITED in the wrong index"},
        {NC_EMAXVARS, "NetCDF: NC_MAX_VARS exceeded"},
        {NC_ENOTVAR, "NetCDF: Variable not found"},
        {NC_EGLOBAL, "NetCDF: Action prohibited on NC_GLOBAL varid"},
        {NC_ENOTNC, "NetCDF: Unknown file format (file format violates CDF specification)"},
        {NC_ESTS, "NetCDF: In Fortran, string too short"},
        {NC_EMAXNAME, "NetCDF: NC_MAX_NAME exceeded"},
        {NC_EUNLIMIT, "NetCDF: NC_UNLIMITED size already in use"},
        {NC_ENORECVARS, "NetCDF: nc_rec op when there are no record vars"},
        {NC_ECHAR, "NetCDF: Attempt to convert between text & numbers"},
        {NC_EEDGE, "NetCDF: Start+count exceeds dimension bound"},
        {NC_ESTRIDE, "NetCDF: Illegal stride"},
        {NC_EBADNAME, "NetCDF: Name contains illegal characters"},
        {NC_ERANGE, "NetCDF: Numeric conversion not representable"},
        {NC_ENOMEM, "NetCDF: Memory allocation (malloc) failure"},
        {NC_EVARSIZE, "NetCDF: One or more variable sizes violate format constraints"},
        {NC_EDIMSIZE, "NetCDF: Invalid dimension size"}
    };

    /* Try on a bad error status */
    message = APIFunc(strerror)(-666);/* should fail */
    expected_msg = "Unknown Error";
    IF (strncmp(message, expected_msg, strlen(expected_msg)) != 0)
        error("APIFunc(strerror) on bad error status returned: %s", message);
    ELSE_NOK

    /* Try on each legitimate error status */
    for (i=0; i<LEN_OF(ncerrs); i++) {
        const char *message = APIFunc(strerror)(ncerrs[i].status);
        IF (strcmp(message, ncerrs[i].msg) != 0)
            error("APIFunc(strerror)(%d) should return \"%s\", not \"%s\"",
                  ncerrs[i].status, ncerrs[i].msg, message);
        ELSE_NOK
    }
    return nok;
}


/*
 * Test APIFunc(open).
 * If in read-only section of tests,
 *    Try to open a non-existent netCDF file, check error return.
 *    Open a file that is not a netCDF file, check error return.
 *    Open a netCDF file with a bad mode argument, check error return.
 *    Open a netCDF file with NC_NOWRITE, read-only mode, try to write, check error.
 *    Try to open a netcdf twice, check whether returned netcdf ids different.
 * If in writable section of tests,
 *    Open a netCDF file with NC_WRITE mode, write something, close it.
 * On exit, any open netCDF files are closed.
 */
#define NOT_NC_FILE "dummy_not_nc_file"
int
TestFunc(open)(void)
{
    int err, ncid, ncid2, nok=0;
ifdef(`PNETCDF', ``#'if 1', ``#'if 0')
    int fd;
#endif

    /* Try to open a nonexistent file */
    err = FileOpen("tooth-fairy.nc", NC_NOWRITE, &ncid); /* should fail */

    /* on some systems (such as Lustre), opening an nonexisting file will
     * actually create the file. In this case, we print the error messages on
     * screen and move on to the next test, instead of aborting the entire test.
     * The created file will be of zero-length and PnetCDF should complain it
     * is not an NC file, i.e. NC_ENOTNC.
     */
    IF (err == NC_NOERR) {
        error("opening a nonexistent file expects to fail, but got NC_NOERR\n");
    }
ifdef(`PNETCDF',
    `else IF (err == NC_ENOTNC) {
        error("opening a nonexistent file actually creates the file, indicating an MPI-IO internal error\n");
    }
    else IF (err != NC_ENOENT && err != NC_EFILE) {
        /* older version of OpenMPI and MPICH may return MPI_ERR_IO instead of
         * MPI_ERR_NO_SUCH_FILE */
        error("expecting NC_ENOENT or NC_EFILE but got %s, indicating an MPI-IO internal error", NC_ERR_CODE_NAME(err));
    }
    else {
        nok++;
    }', `
`#'ifndef USE_PARALLEL
    IF (! NC_ISSYSERR(err))
        error("nc_open of nonexistent file should have returned system error");
`#'endif')

    /* Open a file that is not a netCDF file.  But need a portable
     * test that also works for cross-compiles ... */
    /* err = nc_open("nc_test.o", NC_NOWRITE, &ncid);/\* should fail *\/ */
    /* IF (err != NC_ENOTNC) */
    /*  error("nc_open of non-netCDF file: status = %d", err); */

ifdef(`PNETCDF', ``#'if 1', ``#'if 0')
    /* create a not-nc file */
    fd = open(NOT_NC_FILE, O_CREAT|O_WRONLY, 0600);
    IF (fd == -1) {
        error("Error: creating a non-CDF file (%s)", strerror(errno));
    }
    else {
        ssize_t w_len;
        w_len = write(fd, "0123456789abcdefghijklmnopqrstuvwxyz", 36);
        if (w_len < 0)
            error("Error: writing to a non-CDF file (%s)", strerror(errno));
        close(fd);
    }

    /* Open a file that is not a netCDF file. */
    err = FileOpen(NOT_NC_FILE, NC_NOWRITE, &ncid); /* should fail */
    IF (err != NC_ENOTNC)
        EXPECT_ERR(NC_ENOTNC or NC_EFILE, err)
    ELSE_NOK

    /* delete the not-nc file */
    unlink(NOT_NC_FILE);
#endif

    /* Open a netCDF file in read-only mode, check that write fails */
    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    ELSE_NOK
    err = APIFunc(redef)(ncid);        /* should fail */
    IF (err != NC_EPERM)
        EXPECT_ERR(NC_EPERM, err)
    /* Opened OK, see if can open again and get a different netCDF ID */
    err = FileOpen(testfile, NC_NOWRITE, &ncid2);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    else {
        APIFunc(close)(ncid2);
        nok++;
    }
    IF (ncid2 == ncid)
        error("netCDF IDs for first and second open calls should differ");

    ifdef(`PNETCDF', `if (! read_only)')
    {   /* tests using netCDF scratch file */
        err = FileCreate(scratch, NC_NOCLOBBER, &ncid2);
        IF (err != NC_NOERR)
            error("create: %s", APIFunc(strerror)(err));
        else
            APIFunc(close)(ncid2);
        err = FileOpen(scratch, NC_WRITE, &ncid2);
        IF (err != NC_NOERR)
            error("open: %s", APIFunc(strerror)(err));
        else {
            APIFunc(close)(ncid2);
            nok++;
        }
        err = FileDelete(scratch, info);
        IF (err != NC_NOERR)
            error("remove of %s failed", scratch);
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


/*
 * Test APIFunc(close).
 *    Try to close a netCDF file twice, check whether second close fails.
 *    Try on bad handle, check error return.
 *    Try in define mode and data mode.
 */
int
TestFunc(close)(void)
{
    int ncid, nok=0;
    int err = FileOpen(testfile, NC_NOWRITE, &ncid);

    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    /* Close a netCDF file twice, second time should fail */
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close failed: %s", APIFunc(strerror)(err));
    ELSE_NOK
    err = APIFunc(close)(ncid);
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK

    /* Try with a bad netCDF ID */
    err = APIFunc(close)(BAD_ID);/* should fail */
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK

    /* Close in data mode */
    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close in data mode failed: %s", APIFunc(strerror)(err));
    ELSE_NOK

    ifdef(`PNETCDF', `if (! read_only)')
    {   /* tests using netCDF scratch file */
        err = FileCreate(scratch, NC_NOCLOBBER, &ncid);
        IF (err != NC_NOERR)
            error("create: %s", APIFunc(strerror)(err));
        err = APIFunc(close)(ncid);
        IF (err != NC_NOERR)
            error("close in define mode: %s", APIFunc(strerror)(err));
        ELSE_NOK
        err = FileDelete(scratch, info);
        IF (err != NC_NOERR)
            error("remove of %s failed", scratch);
    }
    return nok;
}


/*
 * Test APIFunc(inq)
 *    Try on bad handle, check error return.
 *    Try in data mode, check returned values.
 *    Try asking for subsets of info.
 * If in writable section of tests,
 *    Try in define mode, after adding an unlimited dimension, variable.
 * On exit, any open netCDF files are closed.
 */
int
TestFunc(inq)(AttVarArgs)
{
    int ncid;
    int ndims;                        /* number of dimensions */
    int nvars;                        /* number of variables */
    int ngatts;                       /* number of global attributes */
    int recdim;                       /* id of unlimited dimension */
    int err;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    /* Try on bad handle */
    err = APIFunc(inq)(BAD_ID, 0, 0, 0, 0);
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK

    err = APIFunc(inq)(ncid, &ndims, &nvars, &ngatts, &recdim);
    IF (err != NC_NOERR)
        error("inq: %s", APIFunc(strerror)(err));
    else IF (ndims != NDIMS)
        error("inq: wrong number of dimensions returned, %d", ndims);
    else IF (nvars != numVars)
        error("inq: wrong number of variables returned, %d", nvars);
    else IF (ngatts != numGatts)
        error("inq: wrong number of global atts returned, %d", ngatts);
    else IF (recdim != RECDIM)
        error("inq: wrong record dimension ID returned, %d", recdim);
    ELSE_NOK

    /* Inguire for no info (useless, but should still work) */
    err = APIFunc(inq)(ncid, 0, 0, 0, 0);
    IF (err != NC_NOERR)
        error("inq for no info failed: %s", APIFunc(strerror)(err));
    ELSE_NOK

    /* Inguire for subsets of info */
    ngatts = numGatts - 1;        /* wipe out previous correct value */
    err = APIFunc(inq)(ncid, 0, 0, &ngatts, 0);
    IF (err != NC_NOERR)
        error("inq for one item failed: %s", APIFunc(strerror)(err));
    else IF (ngatts != numGatts)
        error("inq subset: wrong number of global atts returned, %d", ngatts);
    ELSE_NOK
    ndims = NDIMS - 1;
    nvars = numVars - 1;
    err = APIFunc(inq)(ncid, &ndims, &nvars, 0, 0);
    IF (err != NC_NOERR)
        error("inq for two items failed: %s", APIFunc(strerror)(err));
    else IF (ndims != NDIMS)
        error("inq subset: wrong number of dimensions returned, %d", ndims);
    else IF (nvars != numVars)
        error("inq subset: wrong number of variables returned, %d", nvars);
    ELSE_NOK

    ifdef(`PNETCDF', `if (! read_only)')
    {   /* tests using netCDF scratch file */
        int ncid2;              /* for scratch netCDF dataset */

        err = FileCreate(scratch, NC_NOCLOBBER, &ncid2);
        IF (err != NC_NOERR) {
            error("create: %s", APIFunc(strerror)(err));
        } else {                /* add dim, var, gatt, check inq */
            int ndims0;
            int nvars0;
            int ngatts0;
            int recdim0;
            err = APIFunc(enddef)(ncid2); /* enter data mode */
            err = APIFunc(inq)(ncid2, &ndims0, &nvars0, &ngatts0, &recdim0);
            IF (err != NC_NOERR)
                error("inq: %s", APIFunc(strerror)(err));
            ELSE_NOK
            err = APIFunc(redef)(ncid2); /* enter define mode */
            /* Check that inquire still works in define mode */
            err = APIFunc(inq)(ncid2, &ndims, &nvars, &ngatts, &recdim);
            IF (err != NC_NOERR)
                error("inq in define mode: %s", APIFunc(strerror)(err));
            else IF (ndims != ndims0)
                error("inq in define mode: ndims wrong, %d", ndims);
            else IF (nvars != nvars0)
                error("inq in define mode: nvars wrong, %d", nvars);
            else IF (ngatts != ngatts0)
                error("inq in define mode: ngatts wrong, %d", ngatts);
            else IF (recdim != recdim0)
                error("inq in define mode: recdim wrong, %d", recdim);
            ELSE_NOK

            {
                int did, vid;
                /* Add dim, var, global att */
                err = APIFunc(def_dim)(ncid2, "inqd", 1L, &did);
                IF (err != NC_NOERR)
                    error("def_dim: %s", APIFunc(strerror)(err));
                err = APIFunc(def_var)(ncid2, "inqv", NC_FLOAT, 0, 0, &vid);
                IF (err != NC_NOERR)
                    error("def_var: %s", APIFunc(strerror)(err));
            }
            err = APIFunc(put_att_text)(ncid2, NC_GLOBAL, "inqa", 1+strlen("stuff"),
                                   "stuff");
            IF (err != NC_NOERR)
                error("put_att_text: %s", APIFunc(strerror)(err));

            /* Make sure APIFunc(inq) sees the additions while in define mode */
            err = APIFunc(inq)(ncid2, &ndims, &nvars, &ngatts, &recdim);
            IF (err != NC_NOERR)
                error("inq in define mode: %s", APIFunc(strerror)(err));
            else IF (ndims != ndims0 + 1)
                error("inq in define mode: ndims wrong, %d", ndims);
            else IF (nvars != nvars0 + 1)
                error("inq in define mode: nvars wrong, %d", nvars);
            else IF (ngatts != ngatts0 + 1)
                error("inq in define mode: ngatts wrong, %d", ngatts);
            ELSE_NOK
            err = APIFunc(enddef)(ncid2);
            IF (err != NC_NOERR)
                error("enddef: %s", APIFunc(strerror)(err));

            /* Make sure APIFunc(inq) stills sees additions in data mode */
            err = APIFunc(inq)(ncid2, &ndims, &nvars, &ngatts, &recdim);
            IF (err != NC_NOERR)
                error("inq failed in data mode: %s", APIFunc(strerror)(err));
            else IF (ndims != ndims0 + 1)
                error("inq in define mode: ndims wrong, %d", ndims);
            else IF (nvars != nvars0 + 1)
                error("inq in define mode: nvars wrong, %d", nvars);
            else IF (ngatts != ngatts0 + 1)
                error("inq in define mode: ngatts wrong, %d", ngatts);
            ELSE_NOK
            APIFunc(close)(ncid2);
            err = FileDelete(scratch, info);
            IF (err != NC_NOERR)
                error("remove of %s failed", scratch);
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_natts)(AttArgs)
{
    int ncid;
    int ngatts;                        /* number of global attributes */
    int err, nok=0;

    err = APIFunc(inq_natts)(BAD_ID, &ngatts);
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK
    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    err = APIFunc(inq_natts)(ncid, &ngatts);
    IF (err != NC_NOERR)
        error("inq_natts: %s", APIFunc(strerror)(err));
    else IF (ngatts != numGatts)
        error("inq_natts: wrong number of global atts returned, %d", ngatts);
    ELSE_NOK
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_ndims)(void)
{
    int ncid;
    int ndims;
    int err;
    int nok=0;

    err = APIFunc(inq_ndims)(BAD_ID, &ndims);
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK
    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    err = APIFunc(inq_ndims)(ncid, &ndims);
    IF (err != NC_NOERR)
        error("inq_ndims: %s", APIFunc(strerror)(err));
    else IF (ndims != NDIMS)
        error("inq_ndims: wrong number returned, %d", ndims);
    ELSE_NOK
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_nvars)(VarArgs)
{
    int ncid;
    int nvars;
    int err;
    int nok=0;

    err = APIFunc(inq_nvars)(BAD_ID, &nvars);
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK
    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    err = APIFunc(inq_nvars)(ncid, &nvars);
    IF (err != NC_NOERR)
        error("inq_nvars: %s", APIFunc(strerror)(err));
    else IF (nvars != numVars)
        error("inq_nvars: wrong number returned, %d", nvars);
    ELSE_NOK
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_unlimdim)(void)
{
    int ncid;
    int unlimdim;
    int err;
    int nok=0;

    err = APIFunc(inq_unlimdim)(BAD_ID, &unlimdim);
    IF (err != NC_EBADID)
        EXPECT_ERR(NC_EBADID, err)
    ELSE_NOK
    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    err = APIFunc(inq_unlimdim)(ncid, &unlimdim);
    IF (err != NC_NOERR)
        error("inq_unlimdim: %s", APIFunc(strerror)(err));
    else IF (unlimdim != RECDIM)
        error("inq_unlimdim: wrong number returned, %d", unlimdim);
    ELSE_NOK
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_dimid)(void)
{
    int ncid;
    int dimid;
    int i;
    int err;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    err = APIFunc(inq_dimid)(ncid, "noSuch", &dimid);
    IF (err != NC_EBADDIM)
        EXPECT_ERR(NC_EBADDIM, err)
    ELSE_NOK
    for (i = 0; i < NDIMS; i++) {
        err = APIFunc(inq_dimid)(BAD_ID, dim_name[i], &dimid);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_dimid)(ncid, dim_name[i], &dimid);
        IF (err != NC_NOERR)
            error("inq_dimid for dim_name[%d]=%s : %s",i,dim_name[i], APIFunc(strerror)(err));
        else IF (dimid != i)
            error("expected %d, got %d", i, dimid);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_dim)(void)
{
    int ncid;
    int i;
    int err;
    char name[NC_MAX_NAME];
    IntType length;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < NDIMS; i++) {
        err = APIFunc(inq_dim)(BAD_ID, i, name, &length);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_dim)(ncid, BAD_DIMID, name, &length);
        IF (err != NC_EBADDIM)
            EXPECT_ERR(NC_EBADDIM, err)
        ELSE_NOK
        err = APIFunc(inq_dim)(ncid, i, 0, 0);
        IF (err != NC_NOERR)
            error("inq_dim: %s", APIFunc(strerror)(err));
        ELSE_NOK
        err = APIFunc(inq_dim)(ncid, i, name, &length);
        IF (err != NC_NOERR)
            error("inq_dim: %s", APIFunc(strerror)(err));
        else IF (strcmp(dim_name[i],name))
            error("name expected: %s, got: %s",dim_name[i],name);
        else IF (dim_len[i] != length)
            error("size expected: %d, got: %d",dim_len[i],length);
        ELSE_NOK
        err = APIFunc(inq_dim)(ncid, i, name, 0);
        IF (err != NC_NOERR)
            error("inq_dim: %s", APIFunc(strerror)(err));
        else IF (strcmp(dim_name[i],name))
            error("name expected: %s, got: %s",dim_name[i],name);
        ELSE_NOK
        err = APIFunc(inq_dim)(ncid, i, 0, &length);
        IF (err != NC_NOERR)
            error("inq_dim: %s", APIFunc(strerror)(err));
        else IF (dim_len[i] != length)
            error("size expected: %d, got: %d",dim_len[i],length);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_dimlen)(void)
{
    int ncid;
    int i;
    int err;
    IntType  length;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < NDIMS; i++) {
        err = APIFunc(inq_dimlen)(BAD_ID, i, &length);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_dimlen)(ncid, BAD_DIMID, &length);
        IF (err != NC_EBADDIM)
            EXPECT_ERR(NC_EBADDIM, err)
        ELSE_NOK
        err = APIFunc(inq_dimlen)(ncid, i, &length);
        IF (err != NC_NOERR)
            error("inq_dimlen: %s", APIFunc(strerror)(err));
        else IF (dim_len[i] != length)
            error("size expected: %d, got: %d",dim_len[i],length);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_dimname)(void)
{
    int ncid;
    int i;
    int err;
    char name[NC_MAX_NAME];
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < NDIMS; i++) {
        err = APIFunc(inq_dimname)(BAD_ID, i, name);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_dimname)(ncid, BAD_DIMID, name);
        IF (err != NC_EBADDIM)
            EXPECT_ERR(NC_EBADDIM, err)
        ELSE_NOK
        err = APIFunc(inq_dimname)(ncid, i, name);
        IF (err != NC_NOERR)
            error("inq_dimname: %s", APIFunc(strerror)(err));
        else IF (strcmp(dim_name[i],name))
            error("name expected: %s, got: %s",dim_name[i],name);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_varid)(VarArgs)
{
    int ncid;
    int varid;
    int i;
    int err;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    err = APIFunc(inq_varid)(ncid, "noSuch", &varid);
    IF (err != NC_ENOTVAR)
        EXPECT_ERR(NC_ENOTVAR, err)
    ELSE_NOK

    for (i = 0; i < numVars; i++) {
        err = APIFunc(inq_varid)(BAD_ID, var_name[i], &varid);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_varid)(ncid, var_name[i], &varid);
        IF (err != NC_NOERR)
            error("inq_varid: %s", APIFunc(strerror)(err));
        else IF (varid != i)
            error("expected %d, got %d", i, varid);
        ELSE_NOK
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_var)(VarArgs)
{
    int ncid;
    int i;
    int err;
    char name[NC_MAX_NAME];
    nc_type datatype;
    int ndims;
    int dimids[MAX_RANK];
    int natts;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < numVars; i++) {
        err = APIFunc(inq_var)(BAD_ID, i, name, &datatype, &ndims, dimids, &natts);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_var)(ncid,BAD_VARID,name,&datatype,&ndims,dimids,&natts);
        IF (err != NC_ENOTVAR)
            EXPECT_ERR(NC_ENOTVAR, err)
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, 0, 0, 0, 0, 0);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, name, &datatype, &ndims, dimids, &natts);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        else IF (strcmp(var_name[i],name))
            error("name expected: %s, got: %s",var_name[i],name);
        else IF (var_type[i] != datatype)
            error("type expected: %d, got: %d",var_type[i],datatype);
        else IF (var_rank[i] != ndims)
            error("ndims expected: %d, got: %d",var_rank[i],ndims);
        else IF (!int_vec_eq(var_dimid[i],dimids,ndims))
            error("unexpected dimid");
        else IF (var_natts[i] != natts)
            error("natts expected: %d, got: %d",var_natts[i],natts);
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, name, 0, 0, 0, 0);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        else IF (strcmp(var_name[i],name))
            error("name expected: %s, got: %s",var_name[i],name);
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, 0, &datatype, 0, 0, 0);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        else IF (var_type[i] != datatype)
            error("type expected: %d, got: %d",var_type[i],datatype);
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, 0, 0, &ndims, 0, 0);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        else IF (var_rank[i] != ndims)
            error("ndims expected: %d, got: %d",var_rank[i],ndims);
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, 0, 0, 0, dimids, 0);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        else IF (!int_vec_eq(var_dimid[i],dimids,ndims))
            error("unexpected dimid");
        ELSE_NOK
        err = APIFunc(inq_var)(ncid, i, 0, 0, 0, 0, &natts);
        IF (err != NC_NOERR)
            error("inq_var: %s", APIFunc(strerror)(err));
        else IF (var_natts[i] != natts)
            error("natts expected: %d, got: %d",var_natts[i],natts);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_vardimid)(VarArgs)
{
    int ncid;
    int i;
    int err;
    int dimids[MAX_RANK];
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < numVars; i++) {
        err = APIFunc(inq_vardimid)(BAD_ID, i, dimids);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_vardimid)(ncid, BAD_VARID, dimids);
        IF (err != NC_ENOTVAR)
            EXPECT_ERR(NC_ENOTVAR, err)
        ELSE_NOK
        err = APIFunc(inq_vardimid)(ncid, i, dimids);
        IF (err != NC_NOERR)
            error("inq_vardimid: %s", APIFunc(strerror)(err));
        else IF (!int_vec_eq(var_dimid[i], dimids, var_rank[i]))
            error("unexpected dimid");
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_varname)(VarArgs)
{
    int ncid;
    int i;
    int err;
    char name[NC_MAX_NAME];
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < numVars; i++) {
        err = APIFunc(inq_varname)(BAD_ID, i, name);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        nok++;
        err = APIFunc(inq_varname)(ncid, BAD_VARID, name);
        IF (err != NC_ENOTVAR)
            EXPECT_ERR(NC_ENOTVAR, err)
        ELSE_NOK
        err = APIFunc(inq_varname)(ncid, i, name);
        IF (err != NC_NOERR)
            error("inq_varname: %s", APIFunc(strerror)(err));
        else IF (strcmp(var_name[i],name))
            error("name expected: %s, got: %s",var_name[i],name);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_varnatts)(AttVarArgs)
{
    int ncid;
    int i;
    int err;
    int natts;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = -1; i < numVars; i++) {
        err = APIFunc(inq_varnatts)(BAD_ID, i, &natts);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_varnatts)(ncid, BAD_VARID, &natts);
        IF (err != NC_ENOTVAR)
            EXPECT_ERR(NC_ENOTVAR, err)
        ELSE_NOK
        err = APIFunc(inq_varnatts)(ncid, VARID(i), &natts);
        IF (err != NC_NOERR)
            error("inq_varnatts: %s", APIFunc(strerror)(err));
        else IF (NATTS(i) != natts)
            error("natts expected: %d, got: %d",NATTS(i),natts);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_varndims)(VarArgs)
{
    int ncid;
    int i;
    int err;
    int ndims;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < numVars; i++) {
        err = APIFunc(inq_varndims)(BAD_ID, i, &ndims);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_varndims)(ncid, BAD_VARID, &ndims);
        IF (err != NC_ENOTVAR)
            EXPECT_ERR(NC_ENOTVAR, err)
        ELSE_NOK
        err = APIFunc(inq_varndims)(ncid, i, &ndims);
        IF (err != NC_NOERR)
            error("inq_varndims: %s", APIFunc(strerror)(err));
        else IF (var_rank[i] != ndims)
            error("ndims expected: %d, got: %d",var_rank[i],ndims);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_vartype)(VarArgs)
{
    int ncid;
    int i;
    int err;
    nc_type datatype;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));
    for (i = 0; i < numVars; i++) {
        err = APIFunc(inq_vartype)(BAD_ID, i, &datatype);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK
        err = APIFunc(inq_vartype)(ncid, BAD_VARID, &datatype);
        IF (err != NC_ENOTVAR)
            EXPECT_ERR(NC_ENOTVAR, err)
        ELSE_NOK
        err = APIFunc(inq_vartype)(ncid, i, &datatype);
        IF (err != NC_NOERR)
            error("inq_vartype: %s", APIFunc(strerror)(err));
        else IF (var_type[i] != datatype)
            error("type expected: %d, got: %d", var_type[i], datatype);
        ELSE_NOK
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


/*
 * Test GetVar1
 */
int
TestFunc(get_var1)(VarArgs)
{
    int ncid;
    int i;
    int err;
    double expect;
    int nok = 0;                /* count of valid comparisons */
    double buf[1];              /* (void *) buffer */
    double value[1];
    IntType j, index[MAX_RANK];
    ifdef(`PNETCDF', `MPI_Datatype datatype;')

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR) error("open: %s", APIFunc(strerror)(err));

    /* check if can detect a bad file ID */
    err = GetVar1(BAD_ID, 0, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_EBADID) EXPECT_ERR(NC_EBADID, err)

    /* check if can detect a bad variable ID */
    err = GetVar1(ncid, BAD_VARID, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_ENOTVAR) EXPECT_ERR(NC_ENOTVAR, err)

    for (i = 0; i < numVars; i++) {
        assert(var_rank[i] <= MAX_RANK);
        assert(var_nels[i] <= MAX_NELS);

        /* check if can detect a bad file ID */
        err = GetVar1(BAD_ID, i, NULL, value, 1, MPI_DATATYPE_NULL);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK

        ifdef(`PNETCDF', `datatype = nc_mpi_type(var_type[i]);')

ifdef(`PNETCDF',`dnl
        err = GetVar1(ncid, i, NULL, value, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EINVALCOORDS) {
            EXPECT_ERR(NC_EINVALCOORDS, err)
        }
        ELSE_NOK
')dnl

        /* test NC_EINVALCOORDS */
        for (j = 0; j < var_rank[i]; j++) {
            index[j] = var_shape[i][j];
            err = GetVar1(ncid, i, index, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            index[j] = 0;
        }
        err = GetVar1(ncid, i, index, value, 1, datatype);
        IF (err != NC_NOERR)
            EXPECT_ERR(NC_NOERR, err)
        ELSE_NOK

        /* check if the contents are supposed to be */
        for (j = 0; j < var_nels[i]; j++) {
            err = toMixedBase(j, var_rank[i], var_shape[i], index);
            IF (err != 0) error("error in toMixedBase");
            expect = hash( var_type[i], var_rank[i], index );
            err = GetVar1(ncid, i, index, buf, 1, datatype);
            IF (err != NC_NOERR)
                EXPECT_ERR(NC_NOERR, err)
            ELSE_NOK
            err = nc2dbl( var_type[i], buf, &value[0]);
            IF (err)
                error("error in nc2dbl");
            if (inRange(expect,var_type[i])) {
                IF (!equal2(value[0],expect,var_type[i]))
                    error("expected: %G, got: %G", expect, value[0]);
                ELSE_NOK
            }
        }
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


/*
 * Test GetVara
 * Choose a random point dividing each dim into 2 parts
 * Get 2^rank (nslabs) slabs so defined
 * Each get overwrites buffer, so check after each get.
 */
int
TestFunc(get_vara)(VarArgs)
{
    int ncid, d, i, k, err, nslabs;
    int nok = 0;      /* count of valid comparisons */
    IntType j, nels;
    IntType start[MAX_RANK];
    IntType edge[MAX_RANK];
    IntType index[MAX_RANK];
    IntType mid[MAX_RANK];
    ifdef(`PNETCDF', `MPI_Datatype datatype;')
    double buf[MAX_NELS];        /* (void *) buffer */
    double expect;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    /* check if can detect a bad file ID */
    err = GetVara(BAD_ID, 0, NULL, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_EBADID) EXPECT_ERR(NC_EBADID, err)

    /* check if can detect a bad variable ID */
    err = GetVara(ncid, BAD_VARID, NULL, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_ENOTVAR) EXPECT_ERR(NC_ENOTVAR, err)

    for (i = 0; i < numVars; i++) {
        assert(var_rank[i] <= MAX_RANK);
        assert(var_nels[i] <= MAX_NELS);

        /* check if can detect a bad file ID */
        err = GetVara(BAD_ID, i, NULL, NULL, buf, 1, MPI_DATATYPE_NULL);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK

        ifdef(`PNETCDF', `datatype = nc_mpi_type(var_type[i]);')

        for (j = 0; j < var_rank[i]; j++) {
            start[j] = 0;
            edge[j] = 1;
        }

ifdef(`PNETCDF',`dnl
        err = GetVara(ncid, i, NULL, NULL, buf, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EINVALCOORDS) {
            EXPECT_ERR(NC_EINVALCOORDS, err)
        }
        ELSE_NOK
        err = GetVara(ncid, i, start, NULL, buf, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EEDGE) {
            EXPECT_ERR(NC_EEDGE, err)
        }
        ELSE_NOK
')dnl

        /* test NC_EINVALCOORDS, first when edge[*] > 0 */
        for (j = 0; j < var_rank[i]; j++) {
            index[j] = var_shape[i][j];
            err = GetVara(ncid, i, index, edge, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            index[j] = 0;
            edge[j] = var_shape[i][j] + 1;  /* edge error check */
            err = GetVara(ncid, i, start, edge, buf, 1, datatype);
            IF (err != NC_EEDGE)
                EXPECT_ERR(NC_EEDGE, err)
            ELSE_NOK
            edge[j] = 1;
        }

        /* Check non-scalars for correct error returned even when there is
         * nothing to get (edge[j]==0) */
        for (j = 0; j < var_rank[i]; j++) edge[j] = 0;

        for (j = 0; j < var_rank[i]; j++) {
            if (var_dimid[i][j] == RECDIM) continue; /* skip record dim */
            start[j] = var_shape[i][j];
            err = GetVara(ncid, i, start, edge, buf, 0, datatype);
#ifndef RELAX_COORD_BOUND
            IF (err != NC_EINVALCOORDS) /* not allowed even when edge[j]==0 */
                EXPECT_ERR(NC_EINVALCOORDS, err)
#else
            IF (err != NC_NOERR) /* allowed when edge[j]==0 */
                EXPECT_ERR(NC_NOERR, err)
#endif
            ELSE_NOK
            start[j] = var_shape[i][j]+1;  /* out of boundary check */
            err = GetVara(ncid, i, start, edge, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            start[j] = 0;
        }
        for (j = 0; j < var_rank[i]; j++) edge[j] = 1;

        err = GetVara(ncid, i, start, edge, buf, 1, datatype);
        IF (err != NC_NOERR)
            EXPECT_ERR(NC_NOERR, err)
        ELSE_NOK

        /* Choose a random point dividing each dim into 2 parts */
        /* get 2^rank (nslabs) slabs so defined */
        nslabs = 1;
        for (j = 0; j < var_rank[i]; j++) {
            mid[j] = roll( var_shape[i][j] );
            nslabs *= 2;
        }
        /* bits of k determine whether to get lower or upper part of dim */
        for (k = 0; k < nslabs; k++) {
            nels = 1;
            for (j = 0; j < var_rank[i]; j++) {
                if ((k >> j) & 1) {
                    start[j] = 0;
                    edge[j] = mid[j];
                }else{
                    start[j] = mid[j];
                    edge[j] = var_shape[i][j] - mid[j];
                }
                nels *= edge[j];
            }
            err = GetVara(ncid, i, start, edge, buf, nels, datatype);
            IF (err != NC_NOERR)
                EXPECT_ERR(NC_NOERR, err)
            ELSE_NOK

            for (j = 0; j < nels; j++) {
                double got;
                char *p = (char *) buf;
                p += j * (IntType)sizeof_nctype(var_type[i]);
                err = nc2dbl( var_type[i], p, & got );
                IF (err) error("error in nc2dbl");
                err = toMixedBase(j, var_rank[i], edge, index);
                IF (err != 0) error("error in toMixedBase");
                for (d = 0; d < var_rank[i]; d++)
                    index[d] += start[d];
                expect = hash(var_type[i], var_rank[i], index);
                if (inRange(expect,var_type[i])) {
                    IF (!equal2(got,expect,var_type[i])) {
                        error("buf read not that expected");
                        if (verbose) {
                            error("\n");
                            error("varid: %d, ", i);
                            error("var_name: %s, ", var_name[i]);
                            error("element number: %d ", j);
                            error("expect: %g", expect);
                            error("got: %g", got);
                        }
                    }
                }
            }
        }
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


/*
 * Test GetVars
 * Choose a random point dividing each dim into 2 parts
 * Get 2^rank (nslabs) slabs so defined
 * Each get overwrites buffer, so check after each get.
 */
int
TestFunc(get_vars)(VarArgs)
{
    int ncid;
    int d;
    int i;
    int k;
    int err;
    int nslabs;
    PTRDType nstarts;   /* number of different starts */
    int nok = 0;        /* total count of valid comparisons */
    int n;              /* count of valid comparisons within var */
    IntType j, m, nels;
    IntType start[MAX_RANK];
    IntType edge[MAX_RANK];
    IntType index[MAX_RANK];
    IntType index2[MAX_RANK];
    IntType mid[MAX_RANK];
    IntType count[MAX_RANK];
    IntType sstride[MAX_RANK];
    PTRDType stride[MAX_RANK];
    ifdef(`PNETCDF', `MPI_Datatype datatype;')
    double buf[MAX_NELS];     /* (void *) buffer */
    char *p;                  /* (void *) pointer */
    double expect;
    double got;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    /* check if can detect a bad file ID */
    err = GetVars(BAD_ID, 0, NULL, NULL, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_EBADID) EXPECT_ERR(NC_EBADID, err)

    /* check if can detect a bad variable ID */
    err = GetVars(ncid, BAD_VARID, NULL, NULL, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_ENOTVAR) EXPECT_ERR(NC_ENOTVAR, err)

    for (i = 0; i < numVars; i++) {
        assert(var_rank[i] <= MAX_RANK);
        assert(var_nels[i] <= MAX_NELS);

        /* check if can detect a bad file ID */
        err = GetVars(BAD_ID, i, NULL, NULL, NULL, buf, 1, MPI_DATATYPE_NULL);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK

        ifdef(`PNETCDF', `datatype = nc_mpi_type(var_type[i]);')

        for (j = 0; j < var_rank[i]; j++) {
            start[j] = 0;
            edge[j] = 1;
            stride[j] = 1;
        }

ifdef(`PNETCDF',`dnl
        err = GetVars(ncid, i, NULL, NULL, NULL, buf, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EINVALCOORDS) {
            EXPECT_ERR(NC_EINVALCOORDS, err)
        }
        ELSE_NOK
        err = GetVars(ncid, i, start, NULL, NULL, buf, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EEDGE) {
            EXPECT_ERR(NC_EEDGE, err)
        }
        ELSE_NOK
')dnl

        /* test NC_EINVALCOORDS, first when edge[*] > 0 */
        for (j = 0; j < var_rank[i]; j++) {
            start[j] = var_shape[i][j];
            err = GetVars(ncid, i, start, edge, stride, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            start[j] = 0;
            edge[j] = var_shape[i][j] + 1;
            err = GetVars(ncid, i, start, edge, stride, buf, 1, datatype);
            IF (err != NC_EEDGE)
                EXPECT_ERR(NC_EEDGE, err)
            ELSE_NOK
            edge[j] = 1;
            stride[j] = 0;
            err = GetVars(ncid, i, start, edge, stride, buf, 1, datatype);
            IF (err != NC_ESTRIDE)
                EXPECT_ERR(NC_ESTRIDE, err)
            ELSE_NOK
            stride[j] = 1;
        }
        /* Check non-scalars for correct error returned even when there is
         * nothing to get (edge[j]==0) */
        for (j = 0; j < var_rank[i]; j++) edge[j] = 0;

        for (j = 0; j < var_rank[i]; j++) {
            if (var_dimid[i][j] == RECDIM) continue; /* skip record dim */
            start[j] = var_shape[i][j];
            err = GetVars(ncid, i, start, edge, stride, buf, 0, datatype);
#ifndef RELAX_COORD_BOUND
            IF (err != NC_EINVALCOORDS) /* not allowed even when edge[j]==0 */
                EXPECT_ERR(NC_EINVALCOORDS, err)
#else
            IF (err != NC_NOERR) /* allowed when edge[j]==0 */
                EXPECT_ERR(NC_NOERR, err)
#endif
            ELSE_NOK
            start[j] = var_shape[i][j]+1;  /* out of boundary check */
            err = GetVars(ncid, i, start, edge, stride, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            start[j] = 0;
        }
        for (j = 0; j < var_rank[i]; j++) edge[j] = 1;

        err = GetVars(ncid, i, start, edge, stride, buf, 1, datatype);
        IF (err != NC_NOERR)
            EXPECT_ERR(NC_NOERR, err)
        ELSE_NOK

        /* Choose a random point dividing each dim into 2 parts */
        /* get 2^rank (nslabs) slabs so defined */
        nslabs = 1;
        for (j = 0; j < var_rank[i]; j++) {
            mid[j] = roll( var_shape[i][j] );
            nslabs *= 2;
        }
        /* bits of k determine whether to get lower or upper part of dim */
        /* choose random stride from 1 to edge */
        n = 0;
        for (k = 0; k < nslabs; k++) {
            nstarts = 1;
            for (j = 0; j < var_rank[i]; j++) {
                if ((k >> j) & 1) {
                    start[j] = 0;
                    edge[j] = mid[j];
                }else{
                    start[j] = mid[j];
                    edge[j] = var_shape[i][j] - mid[j];
                }
                sstride[j] = edge[j] > 0 ? 1+roll(edge[j]) : 1;
                stride[j] = (PTRDType)sstride[j];
                nstarts *= stride[j];
            }
            for (m = 0; m < nstarts; m++) {
                err = toMixedBase(m, var_rank[i], sstride, index);
                IF (err != 0) error("error in toMixedBase");
                nels = 1;
                for (j = 0; j < var_rank[i]; j++) {
                    count[j] = 1 + (edge[j] - index[j] - 1) / (IntType)stride[j];
                    nels *= count[j];
                    index[j] += start[j];
                }
                /* Random choice of forward or backward */
/* TODO
                if ( roll(2) ) {
                    for (j = 0; j < var_rank[i]; j++) {
                        index[j] += (count[j] - 1) * (IntType)stride[j];
                        stride[j] = -stride[j];
                    }
                }
 */
                err = GetVars(ncid, i, index, count, stride, buf, nels, datatype);
                IF (err != NC_NOERR)
                    error("%s", APIFunc(strerror)(err));
                ELSE_NOK

                for (j = 0; j < nels; j++) {
                    p = (char *) buf;
                    p += j * (IntType)sizeof_nctype(var_type[i]);
                    err = nc2dbl( var_type[i], p, & got );
                    IF (err != NC_NOERR)
                        error("error in nc2dbl");
                    err = toMixedBase(j, var_rank[i], count, index2);
                    IF (err != 0) error("error in toMixedBase");
                    for (d = 0; d < var_rank[i]; d++)
                        index2[d] = index[d] + index2[d] * (IntType)stride[d];
                    expect = hash(var_type[i], var_rank[i], index2);
                    if (inRange(expect,var_type[i])) {
                        IF (!equal2(got,expect,var_type[i])) {
                            error("buf read not that expected");
                            if (verbose) {
                                error("\n");
                                error("varid: %d, ", i);
                                error("var_name: %s, ", var_name[i]);
                                error("element number: %d ", j);
                                error("expect: %g, ", expect);
                                error("got: %g ", got);
                            }
                        }
                        ELSE_NOK
                    }
                    n++;
                }
            }
        }
        IF (n != var_nels[i]) {
            error("count != nels");
            if (verbose) {
                error("\n");
                error("varid: %d, ", i);
                error("var_name: %s, ", var_name[i]);
                error("count: %d, ", n);
                error("nels: %d ", var_nels[i]);
            }
        }
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


/*
 * Test GetVarm
 * Choose a random point dividing each dim into 2 parts
 * Get 2^rank (nslabs) slabs so defined
 * Choose random stride from 1 to edge
 * Buffer should end up being bit image of external variable.
 * So all gets for a variable store in different elements of buffer
 */
int
TestFunc(get_varm)(VarArgs)
{
    int ncid;
    int i;
    int k;
    int err;
    int nslabs;
    PTRDType nstarts;   /* number of different starts */
    int nok = 0;        /* total count of valid comparisons */
    IntType j, m, nels;
    IntType start[MAX_RANK];
    IntType edge[MAX_RANK];
    IntType index[MAX_RANK];
    IntType mid[MAX_RANK];
    IntType count[MAX_RANK];
    IntType sstride[MAX_RANK];
    PTRDType stride[MAX_RANK];
    PTRDType imap[MAX_RANK];
    PTRDType imap2[MAX_RANK];
    ifdef(`PNETCDF', `MPI_Datatype datatype;')
    double buf[MAX_NELS];        /* (void *) buffer */
    char *p;                     /* (void *) pointer */
    double expect;
    double got;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    /* check if can detect a bad file ID */
    err = GetVarm(BAD_ID, 0, NULL, NULL, NULL, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_EBADID) EXPECT_ERR(NC_EBADID, err)

    /* check if can detect a bad variable ID */
    err = GetVarm(ncid, BAD_VARID, NULL, NULL, NULL, NULL, NULL, 0, MPI_DATATYPE_NULL);
    IF (err != NC_ENOTVAR) EXPECT_ERR(NC_ENOTVAR, err)

    for (i = 0; i < numVars; i++) {
        assert(var_rank[i] <= MAX_RANK);
        assert(var_nels[i] <= MAX_NELS);

        /* check if can detect a bad file ID */
        err = GetVarm(BAD_ID, i, NULL, NULL, NULL, NULL, buf, 1, MPI_DATATYPE_NULL);
        IF (err != NC_EBADID)
            EXPECT_ERR(NC_EBADID, err)
        ELSE_NOK

        ifdef(`PNETCDF', `datatype = nc_mpi_type(var_type[i]);')

        for (j = 0; j < var_rank[i]; j++) {
            start[j] = 0;
            edge[j] = 1;
            stride[j] = 1;
            imap[j] = 1;
        }

ifdef(`PNETCDF',`dnl
        err = GetVarm(ncid, i, NULL, NULL, NULL, NULL, buf, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EINVALCOORDS) {
            EXPECT_ERR(NC_EINVALCOORDS, err)
        }
        ELSE_NOK
        err = GetVarm(ncid, i, start, NULL, NULL, NULL, buf, 1, datatype);
        if (var_rank[i] == 0) { /* scalar variable */
            IF (err != NC_NOERR) EXPECT_ERR(NC_NOERR, err)
        }
        else IF (err != NC_EEDGE) {
            EXPECT_ERR(NC_EEDGE, err)
        }
        ELSE_NOK
')dnl

        /* test NC_EINVALCOORDS, first when edge[*] > 0 */
        for (j = 0; j < var_rank[i]; j++) {
            start[j] = var_shape[i][j];
            err = GetVarm(ncid, i, start, edge, stride, imap, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            start[j] = 0;
            edge[j] = var_shape[i][j] + 1;
            err = GetVarm(ncid, i, start, edge, stride, imap, buf, 1, datatype);
            IF (err != NC_EEDGE)
                EXPECT_ERR(NC_EEDGE, err)
            ELSE_NOK
            edge[j] = 1;
            stride[j] = 0;
            err = GetVarm(ncid, i, start, edge, stride, imap, buf, 1, datatype);
            IF (err != NC_ESTRIDE)
                EXPECT_ERR(NC_ESTRIDE, err)
            ELSE_NOK
            stride[j] = 1;
        }
        /* Check non-scalars for correct error returned even when there is
         * nothing to get (edge[j]==0) */
        for (j = 0; j < var_rank[i]; j++) edge[j] = 0;

        for (j = 0; j < var_rank[i]; j++) {
            if (var_dimid[i][j] == RECDIM) continue; /* skip record dim */
            start[j] = var_shape[i][j];
            err = GetVarm(ncid, i, start, edge, stride, imap, buf, 0, datatype);
#ifndef RELAX_COORD_BOUND
            IF (err != NC_EINVALCOORDS) /* not allowed even when edge[j]==0 */
                EXPECT_ERR(NC_EINVALCOORDS, err)
#else
            IF (err != NC_NOERR) /* allowed when edge[j]==0 */
                EXPECT_ERR(NC_NOERR, err)
#endif
            ELSE_NOK
            start[j] = var_shape[i][j]+1;  /* out of boundary check */
            err = GetVarm(ncid, i, start, edge, stride, imap, buf, 1, datatype);
            IF (err != NC_EINVALCOORDS)
                EXPECT_ERR(NC_EINVALCOORDS, err)
            ELSE_NOK
            start[j] = 0;
        }
        for (j = 0; j < var_rank[i]; j++) edge[j] = 1;

        err = GetVarm(ncid, i, start, edge, stride, imap, buf, 1, datatype);
        IF (err != NC_NOERR)
            EXPECT_ERR(NC_NOERR, err)
        ELSE_NOK

        if (var_rank[i] > 0) {
            int jj = var_rank[i] - 1;
            /* imap[jj] = sizeof_nctype(var_type[i]); */
            imap[jj] = 1; /* in numbers of elements */
            for (; jj > 0; jj--)
                imap[jj-1] = imap[jj] * (PTRDType)var_shape[i][jj];
        }

        /* Choose a random point dividing each dim into 2 parts */
        /* get 2^rank (nslabs) slabs so defined */
        nslabs = 1;
        for (j = 0; j < var_rank[i]; j++) {
            mid[j] = roll( var_shape[i][j] );
            nslabs *= 2;
        }
        /* bits of k determine whether to get lower or upper part of dim */
        /* choose random stride from 1 to edge */
        for (k = 0; k < nslabs; k++) {
            nstarts = 1;
            for (j = 0; j < var_rank[i]; j++) {
                if ((k >> j) & 1) {
                    start[j] = 0;
                    edge[j] = mid[j];
                }else{
                    start[j] = mid[j];
                    edge[j] = var_shape[i][j] - mid[j];
                }
                sstride[j] = edge[j] > 0 ? 1+roll(edge[j]) : 1;
                stride[j] = (PTRDType)sstride[j];
                imap2[j] = imap[j] * stride[j];
                nstarts *= stride[j];
            }
            for (m = 0; m < nstarts; m++) {
                if (var_rank[i] == 0 && i%2 ) {
                    err = GetVarm(ncid, i, NULL, NULL, NULL, NULL, buf, var_nels[i], datatype);
                } else {
                    err = toMixedBase(m, var_rank[i], sstride, index);
                    IF (err != 0) error("error in toMixedBase");
                    nels = 1;
                    for (j = 0; j < var_rank[i]; j++) {
                        count[j] = 1 + (edge[j] - index[j] - 1) / (IntType)stride[j];
                        index[j] += start[j];
                        nels *= count[j];
                    }
                    /* Random choice of forward or backward */
/* TODO
                    if ( roll(2) ) {
                        for (j = 0; j < var_rank[i]; j++) {
                            index[j] += (count[j] - 1) * (IntType)stride[j];
                            stride[j] = -stride[j];
                        }
                    }
 */
                    j = (int)fromMixedBase(var_rank[i], index, var_shape[i]);
                    p = (char *) buf + j * (IntType)sizeof_nctype(var_type[i]);
                    err = GetVarm(ncid, i, index, count, stride, imap2, p, nels, datatype);
                }
                IF (err != NC_NOERR)
                    error("%s", APIFunc(strerror)(err));
                ELSE_NOK
            }
        }
        p = (char *) buf;
        for (j = 0; j < var_nels[i]; j++) {
            err = toMixedBase(j, var_rank[i], var_shape[i], index);
            IF (err != 0) error("error in toMixedBase");
            expect = hash( var_type[i], var_rank[i], index);
            err = nc2dbl( var_type[i], p, & got );
            IF (err != NC_NOERR)
                error("error in nc2dbl");
            if (inRange(expect,var_type[i])) {
                IF (!equal2(got,expect,var_type[i])) {
                    error("buf read not that expected");
                    if (verbose) {
                        error("\n");
                        error("varid: %d, ", i);
                        error("var_name: %s, ", var_name[i]);
                        error("element number: %d ", j);
                        error("expect: %g, ", expect);
                        error("got: %g ", got);
                    }
                }
                ELSE_NOK
            }
            p += sizeof_nctype(var_type[i]);
        }
    }
    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(get_att)(AttVarArgs)
{
    int ncid;
    int i;
    int j;
    IntType k, ndx[1];
    int err;
    double buf[MAX_NELS];        /* (void *) buffer */
    char *p;                     /* (void *) pointer */
    double expect;
    double got;
    int nok = 0;      /* count of valid comparisons */

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    for (i = -1; i < numVars; i++) {
        for (j = 0; j < NATTS(i); j++) {
            err = APIFunc(get_att)(BAD_ID, i, ATT_NAME(i,j), buf);
            IF (err != NC_EBADID)
                EXPECT_ERR(NC_EBADID, err)
            ELSE_NOK
            err = APIFunc(get_att)(ncid, BAD_VARID, ATT_NAME(i,j), buf);
            IF (err != NC_ENOTVAR)
                EXPECT_ERR(NC_ENOTVAR, err)
            ELSE_NOK
            err = APIFunc(get_att)(ncid, i, "noSuch", buf);
            IF (err != NC_ENOTATT)
                EXPECT_ERR(NC_ENOTATT, err)
            ELSE_NOK
            err = APIFunc(get_att)(ncid, i, ATT_NAME(i,j), buf);
            IF (err != NC_NOERR) {
                error("%s", APIFunc(strerror)(err));
            } else {
                nok++;
                for (k = 0; k < ATT_LEN(i,j); k++) {
                    ndx[0] = k;
                    expect = hash(ATT_TYPE(i,j), -1, ndx);
                    p = (char *) buf;
                    p += k * (IntType)sizeof_nctype(ATT_TYPE(i,j));
                    err = nc2dbl( ATT_TYPE(i,j), p, &got );
                    IF (err != NC_NOERR)
                        error("error in nc2dbl");
                    if (inRange(expect,ATT_TYPE(i,j))) {
                        IF (!equal2(got,expect,ATT_TYPE(i,j))) {
                            error("buf read not that expected");
                            if (verbose) {
                                error("\n");
                                error("varid: %d, ", i);
                                error("var_name: %s, ",
                                        i >= 0 ? var_name[i] : "Global");
                                error("att_name: %s, ", ATT_NAME(i,j));
                                error("element number: %d\n", k);
                                error("expect: %-23.16e\n", expect);
                                error("   got: %-23.16e", got);
                            }
                        }
                    }
                }
            }
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_att)(AttVarArgs)
{
    int ncid;
    int i;
    int j;
    int err;
    nc_type t;
    IntType n;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    for (i = -1; i < numVars; i++) {
        for (j = 0; j < NATTS(i); j++) {
            err = APIFunc(inq_att)(BAD_ID, i, ATT_NAME(i,j), &t, &n);
            IF (err != NC_EBADID)
                EXPECT_ERR(NC_EBADID, err)
            ELSE_NOK
            err = APIFunc(inq_att)(ncid, BAD_VARID, ATT_NAME(i,j), &t, &n);
            IF (err != NC_ENOTVAR)
                EXPECT_ERR(NC_ENOTVAR, err)
            ELSE_NOK
            err = APIFunc(inq_att)(ncid, i, "noSuch", &t, &n);
            IF (err != NC_ENOTATT)
                EXPECT_ERR(NC_ENOTATT, err)
            ELSE_NOK
            err = APIFunc(inq_att)(ncid, i, ATT_NAME(i,j), &t, &n);
            IF (err != NC_NOERR) {
                error("%s", APIFunc(strerror)(err));
            } else {
                IF (t != ATT_TYPE(i,j))
                    error("type not that expected");
                ELSE_NOK
                IF (n != ATT_LEN(i,j))
                    error("length not that expected");
                ELSE_NOK
            }
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_attlen)(AttVarArgs)
{
    int ncid;
    int i;
    int j;
    int err;
    IntType len;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    for (i = -1; i < numVars; i++) {
        err = APIFunc(inq_attlen)(ncid, i, "noSuch", &len);
        IF (err != NC_ENOTATT)
            EXPECT_ERR(NC_ENOTATT, err)
        ELSE_NOK
        for (j = 0; j < NATTS(i); j++) {
            err = APIFunc(inq_attlen)(BAD_ID, i, ATT_NAME(i,j), &len);
            IF (err != NC_EBADID)
                EXPECT_ERR(NC_EBADID, err)
            ELSE_NOK
            err = APIFunc(inq_attlen)(ncid, BAD_VARID, ATT_NAME(i,j), &len);
            IF (err != NC_ENOTVAR)
                EXPECT_ERR(NC_ENOTVAR, err)
            ELSE_NOK
            err = APIFunc(inq_attlen)(ncid, i, ATT_NAME(i,j), &len);
            IF (err != NC_NOERR) {
                error("%s", APIFunc(strerror)(err));
            } else {
                IF (len != ATT_LEN(i,j))
                    error("len not that expected");
                ELSE_NOK
            }
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_atttype)(AttVarArgs)
{
    int ncid;
    int i;
    int j;
    int err;
    nc_type datatype;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    for (i = -1; i < numVars; i++) {
        err = APIFunc(inq_atttype)(ncid, i, "noSuch", &datatype);
        IF (err != NC_ENOTATT)
            EXPECT_ERR(NC_ENOTATT, err)
        ELSE_NOK
        for (j = 0; j < NATTS(i); j++) {
            err = APIFunc(inq_atttype)(BAD_ID, i, ATT_NAME(i,j), &datatype);
            IF (err != NC_EBADID)
                EXPECT_ERR(NC_EBADID, err)
            ELSE_NOK
            err = APIFunc(inq_atttype)(ncid, BAD_VARID, ATT_NAME(i,j), &datatype);
            IF (err != NC_ENOTVAR)
                EXPECT_ERR(NC_ENOTVAR, err)
            ELSE_NOK
            err = APIFunc(inq_atttype)(ncid, i, ATT_NAME(i,j), &datatype);
            IF (err != NC_NOERR) {
                error("%s", APIFunc(strerror)(err));
            } else {
                IF (datatype != ATT_TYPE(i,j))
                    error("type not that expected");
                ELSE_NOK
            }
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_attname)(AttVarArgs)
{
    int ncid;
    int i;
    int j;
    int err;
    char name[NC_MAX_NAME];
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    for (i = -1; i < numVars; i++) {
        err = APIFunc(inq_attname)(ncid, i, BAD_ATTNUM, name);
        IF (err != NC_ENOTATT)
            EXPECT_ERR(NC_ENOTATT, err)
        ELSE_NOK
        err = APIFunc(inq_attname)(ncid, i, NATTS(i), name);
        IF (err != NC_ENOTATT)
            EXPECT_ERR(NC_ENOTATT, err)
        ELSE_NOK
        for (j = 0; j < NATTS(i); j++) {
            err = APIFunc(inq_attname)(BAD_ID, i, j, name);
            IF (err != NC_EBADID)
                EXPECT_ERR(NC_EBADID, err)
            ELSE_NOK
            err = APIFunc(inq_attname)(ncid, BAD_VARID, j, name);
            IF (err != NC_ENOTVAR)
                EXPECT_ERR(NC_ENOTVAR, err)
            ELSE_NOK
            err = APIFunc(inq_attname)(ncid, i, j, name);
            IF (err != NC_NOERR) {
                error("%s", APIFunc(strerror)(err));
            } else {
                IF (strcmp(ATT_NAME(i,j), name) != 0)
                    error("name not that expected");
                ELSE_NOK
            }
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}


int
TestFunc(inq_attid)(AttVarArgs)
{
    int ncid;
    int i;
    int j;
    int err;
    int attnum;
    int nok=0;

    err = FileOpen(testfile, NC_NOWRITE, &ncid);
    IF (err != NC_NOERR)
        error("open: %s", APIFunc(strerror)(err));

    for (i = -1; i < numVars; i++) {
        err = APIFunc(inq_attid)(ncid, i, "noSuch", &attnum);
        IF (err != NC_ENOTATT)
            EXPECT_ERR(NC_ENOTATT, err)
        ELSE_NOK
        for (j = 0; j < NATTS(i); j++) {
            err = APIFunc(inq_attid)(BAD_ID, i, ATT_NAME(i,j), &attnum);
            IF (err != NC_EBADID)
                EXPECT_ERR(NC_EBADID, err)
            ELSE_NOK
            err = APIFunc(inq_attid)(ncid, BAD_VARID, ATT_NAME(i,j), &attnum);
            IF (err != NC_ENOTVAR)
                EXPECT_ERR(NC_ENOTVAR, err)
            ELSE_NOK
            err = APIFunc(inq_attid)(ncid, i, ATT_NAME(i,j), &attnum);
            IF (err != NC_NOERR) {
                error("%s", APIFunc(strerror)(err));
            } else {
                IF (attnum != j)
                    error("attnum not that expected");
                ELSE_NOK
            }
        }
    }

    err = APIFunc(close)(ncid);
    IF (err != NC_NOERR)
        error("close: %s", APIFunc(strerror)(err));
    return nok;
}

