# ------------------------------------
# Copyright (c) Microsoft Corporation.
# Licensed under the MIT License.
# ------------------------------------
"""
DESCRIPTION:
    This sample demonstrates how to get a chat completions response from
    the service using a synchronous client. The sample also shows how to 
    set default chat completions configuration in the client constructor,
    which will be applied to all `complete` calls to the service.

    This sample assumes the AI model is hosted on a Serverless API or
    Managed Compute endpoint. For GitHub Models or Azure OpenAI endpoints,
    the client constructor needs to be modified. See package documentation:
    https://github.com/Azure/azure-sdk-for-python/blob/main/sdk/ai/azure-ai-inference/README.md#key-concepts

USAGE:
    python sample_chat_completions_with_defaults.py

    Set these two environment variables before running the sample:
    1) AZURE_AI_CHAT_ENDPOINT - Your endpoint URL, in the form 
        https://<your-deployment-name>.<your-azure-region>.models.ai.azure.com
        where `your-deployment-name` is your unique AI Model deployment name, and
        `your-azure-region` is the Azure region where your model is deployed.
    2) AZURE_AI_CHAT_KEY - Your model key. Keep it secret.
"""


def sample_chat_completions_with_defaults():
    import os

    try:
        endpoint = os.environ["AZURE_AI_CHAT_ENDPOINT"]
        key = os.environ["AZURE_AI_CHAT_KEY"]
    except KeyError:
        print("Missing environment variable 'AZURE_AI_CHAT_ENDPOINT' or 'AZURE_AI_CHAT_KEY'")
        print("Set them before running this sample.")
        exit()

    from azure.ai.inference import ChatCompletionsClient
    from azure.ai.inference.models import SystemMessage, UserMessage
    from azure.core.credentials import AzureKeyCredential

    # Create a client with default chat completions settings
    client = ChatCompletionsClient(
        endpoint=endpoint, credential=AzureKeyCredential(key), temperature=0.5, max_tokens=1000
    )

    # Call the service with the defaults specified above
    response = client.complete(
        messages=[
            SystemMessage("You are a helpful assistant."),
            UserMessage("How many feet are in a mile?"),
        ],
    )

    print(response.choices[0].message.content)

    # You can always override one or more of the defaults for a specific call, as shown here
    response = client.complete(
        messages=[
            SystemMessage("You are a helpful assistant."),
            UserMessage("What's the distance from Earth to the Moon in miles?"),
        ],
        max_tokens=2000,
    )

    print(response.choices[0].message.content)


if __name__ == "__main__":
    sample_chat_completions_with_defaults()
