/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - TinyCAD netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t etinycad_net;

static int tinycad_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "tinycad") == 0)
		return 100;
	return 0;
}

static void tinycad_print_name(FILE* const f, const char* str)
{
	if(strchr(str, '\''))
	{
		/* TinyCAD does not escape, let the user know the output is broken */
		rnd_message(RND_MSG_ERROR, "tinycad: Broken output! Name contains apostrophe (\'), but it cannot be escaped: %s\n", str);
	}

	fputs(str, f);
}

static void tinycad_print_device(FILE* const f, const char* dev)
{
	if(dev)
	{
		/* TinyCAD prints device name only if it was set */
		tinycad_print_name(f, dev);
	}
}

static void tinycad_print_value(FILE* const f, const char* str)
{
	/* TinyCAD does not escape */
	fputs(str, f);
}

static void tinycad_assert_netlist_item(const char* title, const char* str)
{
	if(strpbrk(str, "(,)"))
	{
		/* TinyCAD does not escape, let the user know the output is broken */
		rnd_message(RND_MSG_ERROR, "tinycad: Broken output! %s contains comma (,) or parenthesis, but it cannot be escaped: %s\n", title, str);
	}
}

/* Export comptag and nettag for attributes that have export_name; the user
   wanted these to end up on the netlist as custom tags/attributes/properties */
static void export_custom_attrs(FILE *f, csch_ahdr_t *obj)
{
	htsp_entry_t *e;
	for(e = htsp_first(&obj->attr); e != NULL; e = htsp_next(&obj->attr, e)) {
		csch_attrib_t *a = e->value;
		if ((a->export_name != NULL) && (a->val != NULL)) {

			fprintf(f, "\tOPTION '");
			tinycad_print_name(f, a->export_name);
			fprintf(f, "' = ");
			tinycad_print_value(f, a->val);
			fprintf(f, "\r\n");
		}
	}
}

/* Export abstract components; exports refdes, footprint, value, device */
/* and pinnum-pinname map as series of OPTIONs */
static void tinycad_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e, *p;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);

		if(fp==NULL)
		{
			fp = "unknown_fp";
			rnd_message(RND_MSG_ERROR, "tinycad: no footprint is set for \"%s\"\n", refdes);
		}

		fprintf(f, "COMPONENT '");
		tinycad_print_name(f, refdes);
		fprintf(f, "' = ");
		tinycad_print_device(f, dev);
		fprintf(f, "\r\n");

		fprintf(f, "\tOPTION 'Package' = ");
		tinycad_print_value(f, fp);
		fprintf(f, "\r\n");

		if(val)
		{
			fprintf(f, "\tOPTION 'Value' = ");
			tinycad_print_value(f, val);
			fprintf(f, "\r\n");
		}

		export_custom_attrs(f, &comp->hdr);

		/* Print terminals so that terminal symbolic names can be preserved.
		   (The netlist uses pin number to reference the terminal, symbolic name
		   is just an useful hint) */
		for(p = htsp_first(&comp->ports); p != NULL; p = htsp_next(&comp->ports, p)) {
			const csch_aport_t *port = p->value;
			const char *pinnums = sch_nle_get_pinnum(port);
			const char *pinname = sch_nle_get_alt_attr(&port->hdr.attr, SCH_NLE_ALTATTR_PINNAME);

			if (pinname == NULL)
				continue;

			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				fprintf(f, "\tOPTION '$$_PINNAME ");
				tinycad_print_name(f, my_num);
				fprintf(f, "' = ");
				tinycad_print_value(f, pinname);
				fprintf(f, "\r\n");
			);
		}
	}
}

/* Export abstract nets; exports net's name and a list of refdes-pinnum pairs */
static void tinycad_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		int net_exported = 0; /* net had at least one connection so it was written */

		if (net->hdr.omit) continue;

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "tinycad: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "tinycad: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "tinycad: can't determine port's parent component refdes\n");*/
				continue;
			}


			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					if(net_exported)
					{
						fputc(',', f);
					}
					else
					{
						net_exported = 1;

						fprintf(f, "NET  '");
						tinycad_print_name(f, netname);
						fprintf(f, "' =  ");
					}

					tinycad_assert_netlist_item("Refdes", refdes);
					tinycad_assert_netlist_item("Pin num", my_num);

					fprintf(f, "(%s,%s)", refdes, my_num);
				}
			);
		}

		/* If the net got exported, print the closing new-line */
		if(net_exported)
		{
			fprintf(f, "\r\n");
		}
	}
}


/* Export netlist from the abstract model */
static int tinycad_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "; ====+  Net List %s  +====\r\n\r\n", fn);

	fprintf(f, "; ======+ The component list\r\n\r\n");
	tinycad_export_comps(f, abs);

	fprintf(f, "\r\n\r\n; ======+ The net list\r\n\r\n");
	tinycad_export_nets(f, abs);

	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_tinycad(int ver_needed) { return 0; }

void pplg_uninit_export_tinycad(void)
{
	csch_plug_io_unregister(&etinycad_net);
	rnd_export_remove_opts_by_cookie(tinycad_cookie);
	rnd_hid_remove_hid(&tinycad_hid);
}

int pplg_init_export_tinycad(void)
{
	RND_API_CHK_VER;

	etinycad_net.name = "export to TinyCAD";
	etinycad_net.export_prio = tinycad_export_prio;
	etinycad_net.export_project_abst = tinycad_export_project_abst;
	etinycad_net.ext_export_project = ".net";
	csch_plug_io_register(&etinycad_net);


	rnd_hid_nogui_init(&tinycad_hid);

	tinycad_hid.struct_size = sizeof(rnd_hid_t);
	tinycad_hid.name = "tinycad";
	tinycad_hid.description = "Exports project's TinyCAD netlist";
	tinycad_hid.exporter = 1;

	tinycad_hid.get_export_options = tinycad_get_export_options;
	tinycad_hid.do_export = tinycad_do_export;
	tinycad_hid.parse_arguments = tinycad_parse_arguments;
	tinycad_hid.argument_array = tinycad_values;

	tinycad_hid.usage = tinycad_usage;

	rnd_hid_register_hid(&tinycad_hid);
	rnd_hid_load_defaults(&tinycad_hid, tinycad_options, NUM_OPTIONS);


	return 0;
}

